//   GreKo chess engine
//   (c) 2002-2021 Vladimir Medvedev <vrm@bk.ru>
//   http://greko.su

#include "eval_params.h"
#include "notation.h"
#include "utils.h"

struct FeatureInfo
{
	FeatureInfo(const string& name_, size_t index_, size_t len_) :
		name(name_),
		index(index_),
		len(len_)
	{}

	string name;
	size_t index;
	size_t len;
};
vector<FeatureInfo> g_features;

vector<double> g_w;

size_t Mid_Pawn;
size_t Mid_Knight;
size_t Mid_Bishop;
size_t Mid_Rook;
size_t Mid_Queen;
size_t Mid_King;

size_t End_Pawn;
size_t End_Knight;
size_t End_Bishop;
size_t End_Rook;
size_t End_Queen;
size_t End_King;

size_t NUMBER_OF_FEATURES;
size_t maxTagLength = 0;

void InitFeatures()
{
	size_t index = 0;
	g_features.clear();

#define REGISTER_FEATURE(var, len)                            \
	var = index;                                              \
	g_features.push_back(FeatureInfo(#var, index, len));      \
	index += len;                                             \
	if (strlen(#var) > maxTagLength)                          \
	    maxTagLength = strlen(#var);

	REGISTER_FEATURE(Mid_Pawn, PSQ_SIZE + 1)
	REGISTER_FEATURE(Mid_Knight, PSQ_SIZE + 1)
	REGISTER_FEATURE(Mid_Bishop, PSQ_SIZE + 1)
	REGISTER_FEATURE(Mid_Rook, PSQ_SIZE + 1)
	REGISTER_FEATURE(Mid_Queen, PSQ_SIZE + 1)
	REGISTER_FEATURE(Mid_King, PSQ_SIZE + 1)

	REGISTER_FEATURE(End_Pawn, PSQ_SIZE + 1)
	REGISTER_FEATURE(End_Knight, PSQ_SIZE + 1)
	REGISTER_FEATURE(End_Bishop, PSQ_SIZE + 1)
	REGISTER_FEATURE(End_Rook, PSQ_SIZE + 1)
	REGISTER_FEATURE(End_Queen, PSQ_SIZE + 1)
	REGISTER_FEATURE(End_King, PSQ_SIZE + 1)

	NUMBER_OF_FEATURES = index;

#undef REGISTER_FEATURE
}
////////////////////////////////////////////////////////////////////////////////

string FeatureName(size_t index)
{
	for (size_t i = 0; i < g_features.size(); ++i)
	{
		if (g_features[i].index <= index &&
			index < g_features[i].index + g_features[i].len)
		{
			stringstream ss;
			ss << g_features[i].name <<
				"[" << index - g_features[i].index << "]";
			return ss.str();
		}
	}
	return "<unknown>";
}
////////////////////////////////////////////////////////////////////////////////

void SetDefaultWeights(vector<double>& x)
{

#ifdef PSQ_5
	static const double src[] =
	{
		100, 0, 0, 0, 10, 0, 325, -30, 0, -30, 0, 0, 350, -30, 0, -30, 0, 0,
		500, 0, 0, 0, 20, 0, 900, -20, 0, -20, 0, 0, 0, 50, 0, 50, 0, 0, 100,
		0, 0, 0, 10, 0, 325, -30, 0, -30, 0, 0, 350, -30, 0, -30, 0, 0, 500,
		0, 0, 0, 20, 0, 900, -20, 0, -20, 0, 0, 0, -40, 0, -40, 0, 0
	};
#endif

#ifdef PSQ_12
	static const double src[] =
	{
		39, 2, 15, 7, 13, 0, 128, 47, -2, -31, -48, -53, 0, 94, 3, 24, 30,
		35, -43, 70, 88, 30, 12, -8, -22, -33, 110, 22, 29, 30, 28, 0, 16,
		75, 18, 14, 10, -4, -20, 99, 7, 27, 21, 42, 126, 75, 78, 16, -38, -49,
		-59, -51, 288, 70, 69, 74, 73, 113, 35, 60, 33, 7, 10, 19, 8, 0, 20,
		23, -32, -11, -5, 27, 57, 7, -36, -55, 0, 4, 124, 33, 43, 29, 18, 0,
		85, 43, 3, -8, -6, 7, 0, 318, 46, 73, 94, 104, 51, 47, 64, 87, 75,
		35, 0, -44, 346, 66, 87, 91, 101, 47, 57, 53, 73, 60, 39, 21, -6, 582,
		139, 145, 155, 141, 65, 100, 94, 98, 87, 51, 38, 44, 1033, 248, 260,
		259, 265, 143, 202, 202, 189, 163, 97, 28, 7, 0, -16, 1, 12, 2, 29,
		36, 22, 18, 11, -8, -40, -68
	};
#endif

#ifdef PSQ_16
	static const double src[] =
	{
		100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 120, 60, 0, 0, 0, 0, 0, 325, -14, -2,
		6, 10, 10, 6, -2, -14, -14, -2, 6, 10, 10, 6, -2, -14, 350, -14, -2,
		6, 10, 10, 6, -2, -14, -14, -2, 6, 10, 10, 6, -2, -14, 500, 0, 0, 0,
		0, 0, 0, 0, 0, 20, 20, 0, 0, 0, 0, 0, 0, 900, -14, -2, 6, 10, 10, 6,
		-2, -14, -14, -2, 6, 10, 10, 6, -2, -14, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 120, 60, 0,
		0, 0, 0, 0, 325, -14, -2, 6, 10, 10, 6, -2, -14, -14, -2, 6, 10, 10,
		6, -2, -14, 350, -14, -2, 6, 10, 10, 6, -2, -14, -14, -2, 6, 10, 10,
		6, -2, -14, 500, 0, 0, 0, 0, 0, 0, 0, 0, 20, 20, 0, 0, 0, 0, 0, 0,
		900, -14, -2, 6, 10, 10, 6, -2, -14, -14, -2, 6, 10, 10, 6, -2, -14,
		0, -14, -2, 6, 10, 10, 6, -2, -14, -14, -2, 6, 10, 10, 6, -2, -14
	};

	/*
	static const double src[] =
	{
		51, -14, -11, -13, -4, 10, 28, 45, 10, 0, 136, 49, -2, -29, -48, -53,
		0, 128, -29, 7, 11, 25, 31, 31, 31, 20, -44, 78, 95, 34, 18, -3, -18,
		-29, 145, 8, 16, 21, 20, 18, 19, 23, 17, 4, 19, 76, 23, 19, 15, 1,
		-15, 141, -11, 0, 12, 25, 32, 11, 53, 15, 133, 84, 85, 20, -34, -46,
		-54, -46, 376, 23, 29, 41, 46, 41, 52, 67, 74, 124, 50, 65, 35, 21,
		24, 32, 20, 0, 27, 34, -10, -67, -9, -44, 37, 33, -5, 27, 58, 7, -35,
		-54, 0, 1, 134, 35, 33, 15, 5, 7, 11, 21, 5, 0, 85, 45, 5, -6, -4,
		8, 0, 343, 8, 29, 60, 67, 66, 54, 45, 11, 57, 50, 67, 90, 76, 38, 3,
		-39, 374, 31, 48, 50, 62, 59, 53, 46, 23, 51, 61, 57, 76, 63, 42, 25,
		-3, 630, 81, 86, 91, 83, 70, 86, 67, 62, 71, 104, 100, 105, 93, 59,
		44, 51, 1120, 120, 140, 145, 146, 157, 142, 135, 132, 153, 210, 215,
		205, 168, 106, 39, 20, 0, -4, 3, 10, 18, 0, 11, -9, -29, 28, 36, 22,
		17, 10, -8, -39, -66
	};
	*/
#endif

#ifdef PSQ_64
	static const double src[] =
	{
		90, 0, 0, 0, 0, 0, 0, 0, 0, 160, 171, 180, 148, 211, 72, 36, 54, -13,
		25, 52, 97, 95, 80, 44, 8, -30, 0, 0, 20, 40, 25, 5, -33, -44, -11,
		-3, 15, 17, -4, 0, -44, -41, -4, -10, -23, -12, -28, 40, -17, -60,
		-9, -25, -53, -46, -7, 42, -35, 0, 0, 0, 0, 0, 0, 0, 0, 423, -178,
		-4, -1, -26, 30, -61, 0, -62, -44, -7, 24, 78, 55, 127, -11, 0, -13,
		40, 68, 104, 139, 143, 82, 23, 7, 6, 51, 44, 44, 89, 28, 57, -16, -1,
		4, 24, 32, 31, 18, 0, -52, -11, -1, 0, 14, 11, 28, -18, -67, -22, 0,
		-17, -7, -2, -22, -26, -53, -47, -80, -20, -42, -20, -32, -92, 467,
		-18, -51, -22, 0, -41, -60, -1, 0, -19, -7, 0, -20, 27, 10, -13, -35,
		-32, 44, 24, 67, 73, 121, 64, 24, 0, 1, 32, 50, 51, 25, 17, 23, -14,
		21, 16, 32, 18, 10, -6, 18, -11, 6, 10, 0, -6, 12, 20, 0, -7, 0, 10,
		-11, -9, 0, 15, -57, -56, -39, -26, -22, -27, -38, -26, -4, 635, 86,
		47, 117, 104, 124, 80, 82, 111, 35, 46, 80, 136, 152, 142, 59, 100,
		5, 49, 60, 104, 109, 79, 118, 47, -29, -1, 13, 33, 71, 24, 32, 4, -42,
		-36, -42, -12, 10, -15, 6, -18, -65, -30, -61, -51, -58, -22, -8, -45,
		-61, -41, -46, -39, -33, -41, 5, -99, -38, -23, -24, -12, 0, -15, -47,
		-32, 1892, -18, -1, 28, 61, 53, 22, 9, 37, -17, -37, -6, 17, 35, 64,
		1, 5, -16, 1, -16, 39, 34, 113, 71, 68, -29, -35, 1, -3, 34, 23, 14,
		23, -14, -24, 0, 6, 13, -13, -5, 0, -31, 0, -13, 0, -10, 0, 2, 0, -25,
		-31, 18, 7, 2, 10, 4, -20, -53, -67, -28, 2, -3, -67, -37, -32, 0,
		0, 0, 0, 0, 0, -12, 0, 0, -1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -2, 0,
		0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, -44, 0, 0, -10, -72, 16,
		3, -33, -51, -81, -17, 0, -5, 49, 11, -13, -56, -13, -3, 46, 86, -117,
		121, 80, -62, 67, -8, 125, 132, 125, 0, 0, 0, 0, 0, 0, 0, 0, 133, 145,
		140, 136, 76, 141, 134, 112, 84, 67, 42, 32, 22, 18, 43, 51, 31, 26,
		-8, -10, -14, -19, -1, 13, 19, 3, -14, -18, -26, -23, -2, -1, 3, 1,
		-22, -15, -19, -5, -17, -12, 20, 0, 0, -5, 13, -1, -4, -7, 0, 0, 0,
		0, 0, 0, 0, 0, 229, -41, 0, 4, -34, 11, -11, -16, -29, -1, -21, 15,
		23, 5, 0, 5, -24, 9, 14, 28, 17, 17, 0, 0, -1, 3, 22, 41, 48, 58, 34,
		46, -4, -26, 9, 43, 39, 38, 40, -6, -9, -60, 9, 3, 30, 39, 9, -28,
		-41, 0, -28, -45, 6, 2, -31, -51, 0, 2, -103, -17, -58, 1, -40, -81,
		-30, 269, 0, -5, 11, 0, 3, 0, -2, 0, -23, 37, 5, 26, 7, 25, 20, -11,
		21, 0, 23, -11, 9, 0, 13, 0, 8, 43, 19, 30, 12, 32, 20, -23, -4, -15,
		25, 36, 28, 10, 28, -45, -25, 13, 18, 30, 15, 9, -24, -12, -65, -31,
		-17, -14, -17, -36, -35, -44, -30, -24, -68, -28, -54, -72, -24, -64,
		459, -2, 1, -6, 0, -8, 6, 2, 3, 24, 21, 9, 0, -13, -6, 7, 3, 18, 2,
		2, 0, -10, 9, -22, 0, 29, 6, 17, 15, -9, 2, 8, 0, 3, 4, 32, 14, -4,
		11, 5, -19, -18, -4, 25, 18, 0, 0, -9, -9, -34, -19, -12, -37, -27,
		-11, -25, -43, -24, -13, 2, 3, -13, -4, 6, -75, 488, -4, 0, 14, 20,
		52, 50, 31, 2, -8, 41, 64, 51, 53, 42, 23, 20, 0, 7, 49, 40, 77, 39,
		44, 0, 3, 4, 4, 67, 50, 43, 54, 11, -49, 0, 0, 32, 18, 20, 22, 1, -31,
		-83, 0, -39, 2, -40, -17, -63, -31, 2, -112, -110, -70, -109, -114,
		-113, -52, -60, -141, -182, -100, -111, -86, -14, 0, 0, -18, 7, 13,
		5, -222, -7, -6, 0, -111, -179, 1, -102, -21, 0, 6, 13, 17, 16, 0,
		12, 7, 16, 0, 0, 2, 12, 17, 21, 15, 13, -10, -7, 3, 5, 20, 17, 21,
		5, -2, -20, 1, 6, 14, 32, 22, 3, -16, -15, -2, 8, 14, 7, 7, -9, -44,
		6, -56, -43, -2, -80, -38, -70, -100
	};
#endif

	x.clear();
	x.resize(NUMBER_OF_FEATURES);
	memcpy(&(x[0]), src, NUMBER_OF_FEATURES * sizeof(double));
}
////////////////////////////////////////////////////////////////////////////////

bool ReadWeights(vector<double>& x, const string& file)
{
	x.clear();
	x.resize(NUMBER_OF_FEATURES);

	ifstream ifs(file.c_str());
	if (ifs.good())
	{
		string s;
		while (getline(ifs, s))
		{
			istringstream iss(s);
			string name;
			double value;

			if (!(iss >> name))
				continue;

			for (size_t i = 0; i < g_features.size(); ++i)
			{
				if (g_features[i].name == name)
				{
					size_t index = g_features[i].index;
					size_t offset = 0;
					while (iss >> value)
					{
						x[index + offset] = value;
						++offset;
						if (offset >= g_features[i].len)
							break;
					}
					break;
				}
			}
		}
		return true;
	}
	return false;
}
////////////////////////////////////////////////////////////////////////////////

void WriteWeights(const vector<double>& x, const string& file)
{
	ofstream ofs(file.c_str());
	if (ofs.good())
	{
		for (size_t i = 0; i < g_features.size(); ++i)
		{
			string name = g_features[i].name;
			size_t index = g_features[i].index;
			size_t len = g_features[i].len;

			ofs << setw(maxTagLength + 2) << left << name;
			for (size_t j = 0; j < len; ++j)
				ofs << " " << (int)x[index + j];
			ofs << endl;

			if (i == g_features.size() / 2 - 1)
				ofs << endl;
		}
	}
}
////////////////////////////////////////////////////////////////////////////////
