//  GREKO Chess Engine
//  (c) 2002-2012 Vladimir Medvedev <vrm@bk.ru>
//  http://greko.110mb.com

//  search.h: chess tree search
//  modified: 05-Aug-2012

#pragma once

#include <math.h>
#include "position.h"
#include "utils.h"

enum
{
  HASH_UNKNOWN = 0x00, 
  HASH_ALPHA   = 0x10, 
  HASH_BETA    = 0x20, 
  HASH_EXACT   = 0x40
};

struct HashEntry
{
  I8   _depth;
  EVAL _eval;
  U64  _hash;
  Move _mv;
  U8   _flags;

  U8 Age() const  { return _flags & 0x0f; }
  U8 Type() const { return _flags & 0xf0; }
};

const EVAL SORT_HASH       = 0x40000000;
const EVAL SORT_CAPTURE    = 0x20000000;
const EVAL SORT_MATEKILLER = 0x18000000;
const EVAL SORT_KILLER     = 0x10000000;

class Search
{
public:
  Search() : 
      _mode(IDLE),
      _hash(NULL),
      _hashSize(0),
      _inc(0),
      _knpsLimit(999999),
      _multiPV(1),
      _rootWindow(VAL_P),
      _sd(0), 
      _sn(0), 
      _stHard(2000), 
      _stSoft(2000) {}

  EVAL  AlphaBetaRoot(EVAL alpha, EVAL beta, const int depth);
  EVAL  AlphaBeta(EVAL alpha, EVAL beta, const int depth, int ply, bool isNull);
  EVAL  AlphaBetaQ(EVAL alpha, EVAL beta, int ply, int qply);
  void  CheckInput();
  void  CheckLimits();
  void  ClearHash();
  void  ClearHistory();
  void  ClearKillers();
  void  Epdtest(FILE* psrc, double time_in_seconds, int reps);
  int   GetIncrement() const { return _inc; }
  int   GetSearchTime() { return 1000 * (clock() - _start_time) / CLOCKS_PER_SEC; } // milliseconds
  NODES Perft(int depth);
  void  PrintPV();
  HashEntry* ProbeHash();
  void  RecordHash(Move bestMove, int depth, EVAL eval, U8 type, int ply);
  void  SetHashMB(double mb);
  void  SetIncrement(int inc) { _inc = inc; }
  void  SetKnps(double knps) { _knpsLimit = knps; }
  void  SetLimits(int restMillisec, int sd, NODES sn, int stHard, int stSoft);
  void  SetMultiPV(int n) { _multiPV = n; }
  void  SetNPS(int nps) { _npsLimit = nps; }
  void  StartAnalyze(const Position& pos);
  bool  StartEpd(const std::string& fen, int reps);
  void  StartPerft(const Position& pos, int depth);
  void  StartThinking(Position& pos);
  void  UpdateScores(MoveList& mvlist, Move hashmv, int ply);
  void  UpdateScoresQ(MoveList& mvlist);

  static EVAL SEE(const Position& pos, Move mv);
  static EVAL SEE_Exchange(const Position& pos, FLD to, COLOR side, EVAL currScore, EVAL target, U64 occ);

  void LimitKnps()
  {
    int searchTime = GetSearchTime();
    double knps = (searchTime > 0)? static_cast<double>(_nodes) / searchTime : 0;
    if (!_flag)
    {
      // out("knps=%d\n", knps);
      if  (knps > _knpsLimit)
      {
        int sleepTime = static_cast<int>(_nodes / _knpsLimit) - searchTime;
        if (_stHard)
        {
          if (sleepTime > _stHard - searchTime)
          {
            sleepTime = _stHard - searchTime;
            _flag = true;
          }
        }
        // out("sleepTime=%d\n", sleepTime);
        SleepMilliseconds(sleepTime);
      }
    }
  }

private:

  enum { MAX_PLY = 64 };
  enum { MAX_BRANCH = 128 };
  enum { IDLE, ANALYZE, THINKING, EPDTEST } _mode;

  struct MultiPVEntry
  {
    std::vector<Move> _pv;
    EVAL _score;
    bool _seen;
  };

  bool         _flag;
  HashEntry*   _hash;
  U8           _hashAge;
  long         _hashSize;
  int          _histTry[14][64];
  int          _histSuccess[14][64];
  int          _inc;
  int          _iter;
  EVAL         _iterScore;
  Move         _killers[MAX_PLY];
  double       _knpsLimit;
  MoveList     _lists[MAX_PLY];
  Move         _matekillers[MAX_PLY];
  int          _multiPV;
  MultiPVEntry _multiPVs[MAX_BRANCH];
  NODES        _nodes;
  int          _npsLimit;
  Position     _pos;
  std::vector<Move> _PV[MAX_PLY];
  int          _restMillisec;
  std::vector<Move> _rootPV;
  EVAL         _rootAlpha;
  EVAL         _rootBeta;
  EVAL         _rootWindow;
  clock_t      _start_time;
  int          _sd;
  NODES        _sn;
  int          _stHard;
  int          _stSoft;
};

