//  GREKO Chess Engine
//  (c) 2002-2012 Vladimir Medvedev <vrm@bk.ru>
//  http://greko.110mb.com

//  search.cpp: chess tree search
//  modified: 05-Aug-2012

#include "book.h"
#include "eval.h"
#include "moves.h"
#include "notation.h"
#include "search.h"
#include "utils.h"

extern Position   g_pos;
extern PROTOCOL_T g_protocol;
extern std::list<std::string> g_commandQueue;
extern Book g_book;

inline bool IsUCI() { return g_protocol == UCI; }

EVAL Search::AlphaBetaRoot(EVAL alpha, EVAL beta, const int depth)
{
  _rootAlpha = alpha;
  _rootBeta = beta;

  int ply = 0;
  bool inCheck = _pos.InCheck();
  _PV[ply].clear();
  _nodes++;

  for (int j = 0; j < MAX_BRANCH; ++j)
  {
    _multiPVs[j]._score = -INFINITY_SCORE;
    _multiPVs[j]._pv.clear();
  }

  Move hashMove = 0;
  U8 hashType = HASH_ALPHA;
  HashEntry* pentry = ProbeHash();
  if (pentry)
  {
    hashMove = pentry->_mv;
  }

  MoveList& mvlist = _lists[ply];
  if (inCheck)
    mvlist.GenCheckEvasions(_pos);
  else
    mvlist.GenAllMoves(_pos);
  UpdateScores(mvlist, hashMove, ply);

  EVAL e = 0;
  int legalMoves = 0;
  Move bestMove = 0;

  for (int i = 0; i < mvlist.Size(); ++i)
  {
    Move mv = mvlist.GetNthBest(i);
    if (_pos.MakeMove(mv))
    {
      legalMoves++;
      _histTry[mv.Piece()][mv.To()] += depth;

      if (IsUCI() && GetSearchTime() > 1000)
      {
        out("info currmove %s", MoveToStrLong(mv));
        out(" currmovenumber %d\n", legalMoves);
      }

      int newDepth = depth - 1;
      bool givesCheck = _pos.InCheck();

      //
      //   EXTENSIONS
      //

      if (givesCheck)
        ++newDepth;
      else if (mv.Piece() == PW && (mv.To() / 8) == 1)
        ++newDepth;
      else if (mv.Piece() == PB && (mv.To() / 8) == 6)
        ++newDepth;
      
      if (_multiPV > 1)
        e = -AlphaBeta(-beta - VAL_Q, -alpha + VAL_Q, newDepth, ply + 1, false);
      else
      {
        if (legalMoves == 1)
          e = -AlphaBeta(-beta, -alpha, newDepth, ply + 1, false);
        else
        {
          e = -AlphaBeta(-alpha - 1, -alpha, newDepth, ply + 1, false);
          if (e > alpha && e < beta)
            e = -AlphaBeta(-beta, -alpha, newDepth, ply + 1, false);
        }
      }
      _pos.UnmakeMove();

      if (_flag)
        return alpha;

      if (legalMoves == 1)
      {
        bestMove = mv;
        _PV[ply].clear();
        _PV[ply].push_back(mv);
        _PV[ply].insert(_PV[ply].end(), _PV[ply + 1].begin(), _PV[ply + 1].end());
      }

      //
      //   Update multipv
      //

      if (legalMoves < MAX_BRANCH)
      {
        MultiPVEntry *mpv = &(_multiPVs[legalMoves - 1]);
        
        mpv->_pv.clear();
        mpv->_score = e;
        mpv->_pv.push_back(mv);
        mpv->_pv.insert(mpv->_pv.end(), _PV[ply + 1].begin(), _PV[ply + 1].end());
      }

      if (e > alpha)
      {
        bestMove = mv;
        if (!mv.Captured())
          _histSuccess[mv.Piece()][mv.To()] += depth;
        hashType = HASH_EXACT;

        _PV[ply].clear();
        _PV[ply].push_back(mv);
        _PV[ply].insert(_PV[ply].end(), _PV[ply + 1].begin(), _PV[ply + 1].end());

        alpha = e;

        if (alpha >= beta)
        {
          hashType = HASH_BETA;
          if (!mv.Captured())
          {
            if (e > CHECKMATE_SCORE - MAX_PLY && e <= CHECKMATE_SCORE)
              _matekillers[ply] = mv;
            else
              _killers[ply] = mv;              
          }
          break;
        }
      }
    }
  }
  if (legalMoves == 0)
  {
    if (inCheck)
      alpha = -CHECKMATE_SCORE + ply;
    else
      alpha = DRAW_SCORE;
  }
  RecordHash(bestMove, depth, alpha, hashType, ply);
  return alpha;
}

EVAL Search::AlphaBeta(EVAL alpha, EVAL beta, const int depth, int ply, bool isNull)
{
  _PV[ply].clear();
  ++_nodes;

  COLOR side = _pos.Side();
  bool onPV = (beta - alpha > 1);
  bool inCheck = _pos.InCheck();
  bool lateEndgame = _pos.MatIndex(side) < 5;

  CheckLimits();
  if (_nodes % 8192 == 0)
    CheckInput();

  //
  //   DRAW DETECTION
  //

  if (ply >= MAX_PLY) return DRAW_SCORE;
  EVAL_ESTIMATION ee = EstimateDraw(_pos);
  if (ee == EVAL_THEORETICAL_DRAW || ee == EVAL_PRACTICAL_DRAW) return DRAW_SCORE;

  if (!isNull)
  {
    int rep_total = _pos.GetRepetitions();
    if (rep_total >= 2)
      return DRAW_SCORE;
  }

  //
  //   PROBING HASH
  //

  Move hashMove = 0;
  U8 hashType = HASH_ALPHA;
  HashEntry* pentry = ProbeHash();

  if (pentry)
  {
    hashMove = pentry->_mv;
    if (pentry->_depth >= depth)
    {
      EVAL hash_eval = pentry->_eval;
      if (hash_eval > CHECKMATE_SCORE - 50 && hash_eval <= CHECKMATE_SCORE)
        hash_eval -= ply;
      else if (hash_eval < -CHECKMATE_SCORE + 50 && hash_eval >= -CHECKMATE_SCORE)
        hash_eval += ply;

      if (pentry->Type() == HASH_EXACT)
        return hash_eval;      
      else if (pentry->Type() == HASH_ALPHA && hash_eval <= alpha)
        return alpha;
      else if (pentry->Type() == HASH_BETA && hash_eval >= beta)
        return beta;
    }
  }

  //
  //   QSEARCH
  //

  if (depth <= 0 && !inCheck)
  {
    --_nodes;
    return AlphaBetaQ(alpha, beta, ply, 0);
  }

  //
  //   PRUNING
  //

  EVAL MARGIN[4] = { INFINITY_SCORE, VAL_P - 50, VAL_B - 50, VAL_R - 50 };
  if (!inCheck && !onPV && depth > 0 && depth < 4)
  {
    EVAL staticScore = Evaluate(_pos, alpha - MARGIN[depth], beta + MARGIN[depth]);
    if (staticScore <= alpha - MARGIN[depth])
      return AlphaBetaQ(alpha, beta, ply, 0);
    if (staticScore >= beta + MARGIN[depth])
      return beta;
  }

  //
  //   NULL MOVE
  //

  const int R = 3;
  if (!inCheck && !isNull && !lateEndgame && depth > 2)
  {
    _pos.MakeNullMove();
    EVAL nullEval;
    nullEval = -AlphaBeta(-beta, -beta + 1, depth - 1 - R, ply + 1, true);
    _pos.UnmakeNullMove();

    if (nullEval >= beta)
      return beta;
  }

  MoveList& mvlist = _lists[ply];
  if (inCheck)
    mvlist.GenCheckEvasions(_pos);
  else
    mvlist.GenAllMoves(_pos);
  UpdateScores(mvlist, hashMove, ply);

  int legalMoves = 0;
  EVAL e = 0;
  Move bestMove = 0;

  for (int i = 0; i < mvlist.Size(); ++i)
  {
    Move mv = mvlist.GetNthBest(i);
    if (_pos.MakeMove(mv))
    {
      legalMoves++;
      _histTry[mv.Piece()][mv.To()] += depth;

      int newDepth = depth - 1;
      bool givesCheck = _pos.InCheck();

      //
      //   EXTENSIONS
      //

      if (givesCheck)
        ++newDepth;
      else if (mv.Piece() == PW && (mv.To() / 8) == 1)
        ++newDepth;
      else if (mv.Piece() == PB && (mv.To() / 8) == 6)
        ++newDepth;

      //
      //   LMR
      //

      bool reduced = false;
      if (!onPV && !inCheck && !givesCheck && !mv.Captured() && !mv.Promotion())
      {
        if (legalMoves > 3 && depth > 2)
        {
          if (_histSuccess[mv.Piece()][mv.To()] <= _histTry[mv.Piece()][mv.To()] / 2)
          {
            --newDepth;
            reduced = true;
          }
        }
      }
      
      if (legalMoves == 1)
        e = -AlphaBeta(-beta, -alpha, newDepth, ply + 1, false);
      else
      {
        e = -AlphaBeta(-alpha - 1, -alpha, newDepth, ply + 1, false);
        if (reduced && e > alpha)
        {
          ++newDepth;
          e = -AlphaBeta(-alpha - 1, -alpha, newDepth, ply + 1, false);
        }
        if (e > alpha && e < beta)
          e = -AlphaBeta(-beta, -alpha, newDepth, ply + 1, false);
      }
      _pos.UnmakeMove();

      if (_flag)
        return alpha;

      if (e > alpha)
      {
        bestMove = mv;
        if (!mv.Captured())
          _histSuccess[mv.Piece()][mv.To()] += depth;
        hashType = HASH_EXACT;

        alpha = e;
        if (alpha >= beta)
        {
          hashType = HASH_BETA;
          if (!mv.Captured())
          {
            if (e > CHECKMATE_SCORE - MAX_PLY && e <= CHECKMATE_SCORE)
              _matekillers[ply] = mv;
            else
              _killers[ply] = mv;
          }

          break;
        }

        _PV[ply].clear();
        _PV[ply].push_back(mv);
        _PV[ply].insert(_PV[ply].end(), _PV[ply + 1].begin(), _PV[ply + 1].end());
      }
    }
  }
  if (legalMoves == 0)
  {
    if (inCheck)
      alpha = -CHECKMATE_SCORE + ply;
    else
      alpha = DRAW_SCORE;
  }
  else if (_pos.Fifty() >= 100)
    alpha = DRAW_SCORE;

  RecordHash(bestMove, depth, alpha, hashType, ply);
  return alpha;
}

EVAL Search::AlphaBetaQ(EVAL alpha, EVAL beta, int ply, int qply)
{
  _PV[ply].clear();
  ++_nodes;

  CheckLimits();
  if (_nodes % 8192 == 0)
    CheckInput();

  if (ply >= MAX_PLY)
    return DRAW_SCORE;

  bool inCheck = _pos.InCheck();
  EVAL staticEval = Evaluate(_pos, alpha, beta);
  if (!inCheck)
  {
    if (staticEval > alpha)
      alpha = staticEval;
  }

  if (alpha >= beta)
    return beta;

  MoveList& mvlist = _lists[ply];
  if (inCheck)
    mvlist.GenCheckEvasions(_pos);
  else
    mvlist.GenCaptures(_pos, qply < 2);
  UpdateScoresQ(mvlist);

  EVAL e = 0;
  int legalMoves = 0;
  for (int i = 0; i < mvlist.Size(); ++i)
  {
    Move mv = mvlist.GetNthBest(i);    
    if (!inCheck && SEE(_pos, mv) < 0)
      continue;

    if (_pos.MakeMove(mv))
    {
      legalMoves++;
      e = -AlphaBetaQ(-beta, -alpha, ply + 1, qply + 1);
      _pos.UnmakeMove();

      if (_flag)
        return alpha;

      if (e > alpha)
      {
        alpha = e;
        if (alpha >= beta)
          break;

        _PV[ply].clear();
        _PV[ply].push_back(mv);
        _PV[ply].insert(_PV[ply].end(), _PV[ply + 1].begin(), _PV[ply + 1].end());
      }
    }
  }

  if (inCheck && legalMoves == 0)
    alpha = -CHECKMATE_SCORE + ply;

  return alpha;
}

void Search::CheckInput()
{
  if (_rootPV.empty())
    return;

  if (InputAvailable())
  {
    char buf[4096];
    ReadInput(buf, sizeof(buf));
    TokenString ts(buf);
    std::string cmd = ts.GetToken();

    if (_mode == ANALYZE)
    {
      Move mv = StrToMove(cmd, g_pos);
      if (mv)
      {
        _flag = true;
        g_commandQueue.push_back("force");
        g_commandQueue.push_back(ts.Str());
        g_commandQueue.push_back("analyze");
      }
      else if (Is(cmd, "board", 1))
        g_pos.Print();
      else if (Is(cmd, "quit", 1))
        exit(0);
      else if (Is(cmd, "isready", 7))
        out("readyok\n");
      else if (Is(cmd, "exit", 2))
        _flag = true;
      else if (Is(cmd, "stop", 2))
        _flag = true;
      else if (Is(cmd, "new", 3))
        _flag = true;
      else if (Is(cmd, "setoption", 8))
      {
        ts.GetToken(); // "name"
        std::string token = ts.GetToken(); // name
        if (token == "MultiPV")
        {
          ts.GetToken(); // "value"
          token = ts.GetToken(); // value
          _multiPV = atoi(token.c_str());
        }
      }
      else if (Is(cmd, "undo", 4))
      {
        _flag = true;
        g_commandQueue.push_back("undo");
        g_commandQueue.push_back("analyze");
      }
      else if (Is(cmd, "remove", 6))
      {
        _flag = true;
        g_commandQueue.push_back("undo");
        g_commandQueue.push_back("analyze");
      }
    }

    else if (_mode == THINKING)
    {
      if (Is(cmd, "quit", 1))
        exit(0);
      else if (Is(cmd, "isready", 7))
        out("readyok\n");
      else if (Is(cmd, "?", 1))
        _flag = true;
      else if (Is(cmd, "stop", 4))
        _flag = true;
      else if (Is(cmd, "new", 3))
        _flag = true;
      else if (Is(cmd, "result", 6))
        _flag = true;
      else if (Is(cmd, "setoption", 8))
      {
        ts.GetToken(); // "name"
        std::string token = ts.GetToken(); // name
        if (token == "MultiPV")
        {
          ts.GetToken(); // "value"
          token = ts.GetToken(); // value
          _multiPV = atoi(token.c_str());
        }
      }
    }

    else if (_mode == EPDTEST)
    {
      if (Is(cmd, "board", 1))
        g_pos.Print();
      else if (Is(cmd, "quit", 1))
        exit(0);
      else if (Is(cmd, "exit", 2))
        _flag = true;
      else if (Is(cmd, "new", 3))
        _flag = true;
    }
  }
}

void Search::CheckLimits()
{
  if (_rootPV.empty())
    return;

  if (_stHard)
  {
    if (GetSearchTime() >= _stHard)
    {
      if (_mode == THINKING || _mode == EPDTEST)
        _flag = true;
    }
  }

  if (_sn && _nodes >= _sn)
  {
    if (_mode == THINKING || _mode == EPDTEST)
      _flag = true;
  }
}

void Search::ClearHash()
{
  memset(_hash, 0, _hashSize * sizeof(HashEntry));
}

void Search::ClearHistory()
{  
  memset(_histTry, 0, 64 * 14 * sizeof(_histTry[0][0]));
  memset(_histSuccess, 0, 64 * 14 * sizeof(_histSuccess[0][0]));
}

void Search::ClearKillers()
{
  memset(_killers, 0, MAX_PLY * sizeof(Move));
  memset(_matekillers, 0, MAX_PLY * sizeof(Move));
}

void Search::Epdtest(FILE* src, double seconds, int reps)
{
  char fen[256];
  int dt = static_cast<int>(1000 * seconds);
  SetLimits(0, 0, 0, dt, dt);
  int total = 0, solved = 0;

  while (fgets(fen, sizeof(fen), src))
  {
    if (_pos.SetFen(fen) == false)
      continue;

    out(fen);
    out("\n");

    ++total;
    if (StartEpd(fen, reps))
      ++solved;

    out("\nScore: %d / %d\n\n", solved, total);
  }
}

NODES Search::Perft(int depth)
{
  if (depth == 0)
    return 1;

  MoveList& mvlist = _lists[depth];
  mvlist.GenAllMoves(_pos);
  NODES sum = 0, delta = 0;

  for (int i = 0; i < mvlist.Size(); ++i)
  {
    Move mv = mvlist[i]._mv;
    if (_pos.MakeMove(mv))
    {
      delta = Perft(depth - 1);
      sum += delta;
      _pos.UnmakeMove();
    }
  }

  return sum;
}

void Search::PrintPV()
{
  Move mv = 0;

  if (IsUCI())
  {
    for (int j = 0; j < MAX_BRANCH; ++j)
      _multiPVs[j]._seen = false;

    for (int j = 1; j <= _multiPV; ++j)
    {
      EVAL best_score = -INFINITY_SCORE;
      MultiPVEntry *best_mpv = NULL;  
      MultiPVEntry *mpv = NULL;

      for (int k = 0; k < MAX_BRANCH; ++k)
      {
        mpv = &(_multiPVs[k]);
        if (mpv->_seen)
          continue;
        if (mpv->_pv.empty())
          break;
        if (mpv->_score > best_score)
        {
          best_score = mpv->_score;
          best_mpv = mpv;
        }
      }

      if (best_mpv)
      {
        best_mpv->_seen = true;
        mpv = best_mpv;
        out("info multipv %d depth %d score", j, _iter);

        if (mpv->_score > CHECKMATE_SCORE - 50)
          out(" mate %d", 1 + (CHECKMATE_SCORE - mpv->_score) / 2);
        else if (mpv->_score < -CHECKMATE_SCORE + 50)
          out(" mate -%d", (mpv->_score + CHECKMATE_SCORE) / 2);
        else
          out(" cp %d", mpv->_score);

        out(" time %ld", GetSearchTime());
        out(" nodes %ld", _nodes);
        if (GetSearchTime() > 0)
        {
          double knps = static_cast<double>(_nodes) / GetSearchTime();
          out (" nps %d", static_cast<int>(1000 * knps));
        }
        out(" pv ");
        for (size_t m = 0; m < mpv->_pv.size(); ++m)
        {
          mv = mpv->_pv[m];
          out("%s ", MoveToStrLong(mv));
        }
        out("\n");
      }
    }
    if (GetSearchTime() > 0)
    {
      out("info time %ld", GetSearchTime());
      out(" nodes %ld", _nodes);
      double knps = static_cast<double>(_nodes) / GetSearchTime();
      out (" nps %d", static_cast<int>(1000 * knps));
    }
    out("\n");
    return;
  }
  else
  {
    out(" %2d", _iter);
    out(" %9d ", _iterScore);
    out(" %7d ", GetSearchTime() / 10);
    out(" %12d ", _nodes);
    out(" ");
  }

  Position tmp = _pos;
  int movenum = tmp.Ply() / 2 + 1;
  for (size_t m = 0; m < _rootPV.size(); ++m)
  {
    mv = _rootPV[m];
    if (tmp.Side() == WHITE)
    {
      out("%d. ", movenum++);
    }
    else if (m == 0)
    {
      out("%d. ... ", movenum++);
    }

    MoveList mvlist;
    mvlist.GenAllMoves(tmp);
    out("%s", MoveToStrShort(mv, tmp));
    tmp.MakeMove(mv);

    if (tmp.InCheck())
    {
      if (int(_iterScore + m + 1) == CHECKMATE_SCORE || int(_iterScore - m - 1) == -CHECKMATE_SCORE)
      {
        out("#");
        if (_iterScore > 0)
          out(" {+");
        else
          out(" {-");
        out("Mate in %d}", m / 2 + 1);
      }
      else
        out("+");
    }
    if (m == 0)
    {
      if (_iterScore <= _rootAlpha)
        out("?");
      else if (_iterScore >= _rootBeta)
        out("!");
    }
    out(" ");
  }
  out("\n");
}

HashEntry* Search::ProbeHash()
{
  long index = long(_pos.Hash() % _hashSize);
  if (_hash[index]._hash == _pos.Hash())
    return _hash + index;
  else
    return NULL;
}

void Search::RecordHash(Move bestMove, int depth, EVAL eval, U8 hashType, int ply)
{
  long index = long(_pos.Hash() % _hashSize);
  HashEntry* pentry = _hash + index;

  if (depth > pentry->_depth || _hashAge != pentry->Age())
  {
    if (eval > CHECKMATE_SCORE - 50 && eval <= CHECKMATE_SCORE)
      eval += ply;
    else if (eval < -CHECKMATE_SCORE + 50 && eval >= -CHECKMATE_SCORE)
      eval -= ply;

    pentry->_depth = static_cast<I8>(depth);
    pentry->_eval = eval;
    pentry->_hash = _pos.Hash();
    pentry->_mv = bestMove;
    pentry->_flags = hashType | _hashAge;
  }
}

void Search::SetHashMB(double mb)
{
  if (_hash)
    delete[] _hash;

  _hashSize = long (1024 * 1024 * mb / sizeof(HashEntry));
  if (_hashSize <= 0)
    _hashSize = 1;

  _hash = new HashEntry[_hashSize];

  if (!IsUCI())
  {
    out("main hash: %8ld nodes = ", _hashSize);
    out("%f MB\n", _hashSize * sizeof(HashEntry) / (1024. * 1024.));
  }
}

void Search::SetLimits(int restMillisec, int sd, NODES sn, int stHard, int stSoft)
{
  _restMillisec = restMillisec;
  _sd = sd;
  _sn = sn;
  _stHard = stHard;
  _stSoft = stSoft;
}

void Search::StartAnalyze(const Position& pos)
{
  _start_time = clock();
  _pos = pos;
  _hashAge = (_hashAge + 1) & 0x0f;
  _mode = ANALYZE;
  _flag = false;
  _nodes = 0;

  SetLimits(0, 0, 0, 0, 0);

  ClearHash();
  ClearHistory();
  ClearKillers();

  if (!IsUCI())
  {
    out("\n%s\n\n", _pos.Fen());
  }

  EVAL alpha = -INFINITY_SCORE;
  EVAL beta = INFINITY_SCORE;

  for (_iter = 1; _iter < MAX_PLY; _iter++)
  {
    EVAL e = AlphaBetaRoot(alpha, beta, _iter);
    LimitKnps();

    if (_flag)
      break;

    _iterScore = e;
    if (e > alpha)
      _rootPV = _PV[0];

    if (e > alpha && e < beta)
    {
      PrintPV();

      alpha = e - _rootWindow / 2;
      beta = e + _rootWindow / 2;
    }
    else
    {
      PrintPV();

      alpha = -INFINITY_SCORE;
      beta = INFINITY_SCORE;
      _iter--;
    }
  }
}

bool Search::StartEpd(const std::string& fen, int reps)
{
  _hashAge = (_hashAge + 1) & 0x0f;
  _mode = EPDTEST;
  _flag = false;
  _nodes = 0;
  int counter = 0;  

  ClearHash();
  ClearHistory();
  ClearKillers();

   _start_time = clock();

  EVAL alpha = -INFINITY_SCORE;
  EVAL beta = INFINITY_SCORE;

  for (_iter = 1; _iter < MAX_PLY; _iter++)
  {
    bool moveFound = false;
    EVAL e = AlphaBetaRoot(alpha, beta, _iter);
    LimitKnps();

    if (_flag)
      break;

    _iterScore = e;
    if (e > alpha)
      _rootPV = _PV[0];

    if (!_rootPV.empty())
    {
      Move mv = _rootPV[0];
      std::string mvstr = MoveToStrShort(mv, _pos);

      if (fen.find(mvstr) != std::string::npos)
        moveFound = true;
      else
      {
        mvstr = MoveToStrLong(mv);
        if (fen.find(mvstr) != std::string::npos)
          moveFound = true;
      }

      if (fen.find("am") != std::string::npos && fen.find("bm") == std::string::npos)
        moveFound = !moveFound;
    }

    if (e > alpha && e < beta)
    {
      alpha = e - _rootWindow / 2;
      beta = e + _rootWindow / 2;
    }
    else
    {
      alpha = -INFINITY_SCORE;
      beta = INFINITY_SCORE;
      _iter--;
    }

    if (moveFound)
    {
      ++counter;
      Highlight(true);
      out(" yes ");
      Highlight(false);
    }
    else
    {
      counter = 0;
      out("  no ");
    }

    Highlight(counter >= reps);
    PrintPV();
    Highlight(false);

    if (counter >= reps)
      break;
  }

  return (counter > 0);
}

void Search::StartThinking(Position& pos)
{
  _start_time = clock();

  std::string message;
  if (pos.IsGameOver(message))
  {
    out("%s\n", message);
    return;
  }

  _pos = pos;
  _hashAge = (_hashAge + 1) & 0x0f;
  _rootPV.clear();
  EVAL e = 0;

  _mode = THINKING;
  _flag = false;
  _nodes = 0;

  // ClearHash();
  ClearHistory();
  ClearKillers();

  if (!IsUCI())
  {
    out("\n%s\n\n", _pos.Fen());
  }

  EVAL alpha = -INFINITY_SCORE;
  EVAL beta = INFINITY_SCORE;
  Move best_move = 0;

  // Book

  std::string buf;
  Move book_move = g_book.GetMove(pos, buf);
  if (book_move)
  {
    best_move = book_move;
    if (!IsUCI())
      out(" 0 0 0 0      (%s)", buf);
    _flag = true;
    goto MAKE_MOVE;
  }

  for (_iter = 1; _iter < MAX_PLY; ++_iter)
  {
    e = AlphaBetaRoot(alpha, beta, _iter);
    LimitKnps();

    if (_flag)
      break;

    _iterScore = e;
    if (e > alpha)
      _rootPV = _PV[0];

    if (e > alpha && e < beta)
    {
      alpha = e - _rootWindow / 2;
      beta = e + _rootWindow / 2;

      if (_stSoft)
      {
        if (GetSearchTime() >= _stSoft)
          _flag = true;
      }
      if (_sd)
      {
        if (_iter >= _sd)
          _flag = true;
      }
    }
    else
    {
      // Additional time
      if (_restMillisec)
      {
        if (_restMillisec > 10 * _stHard)
        {
          int delta = _stHard / 2;
          _stHard += delta;
          _stSoft += delta;
        }
      }
 
      // Opening window
      if (e <= alpha)
        alpha = -INFINITY_SCORE;
      else
        beta = INFINITY_SCORE;
      --_iter;
    }

    PrintPV();
    best_move = _rootPV[0];

    if (best_move && (_iter + e >= CHECKMATE_SCORE))
      _flag = true;
  }

  if (_iter == MAX_PLY && best_move)
    _flag = true;

MAKE_MOVE:

  if (_flag)
  {    
    if (IsUCI())
      out("\nbestmove %s\n", MoveToStrLong(best_move));
    else
    {
      Highlight(true);
      out("\nmove %s\n", MoveToStrLong(best_move));
      Highlight(false);
      pos.MakeMove(best_move);
      if (pos.IsGameOver(message))
        out("%s\n", message);
      out("\n");
    }
  }
}

void Search::StartPerft(const Position& pos, int depth)
{
  _pos = pos;
  int t0 = clock();
  NODES sum = 0, delta = 0;

  MoveList& mvlist = _lists[depth];
  mvlist.GenAllMoves(pos);

  out("\n");
  int i = 0;
  for (i = 0; i < mvlist.Size(); ++i)
  {
    Move mv = mvlist[i]._mv;

    if (_pos.MakeMove(mv))
    {
      delta = Perft(depth - 1);
      sum += delta;
      _pos.UnmakeMove();

      out(" %s - ", MoveToStrLong(mv));
      out("%d\n", delta);
      if (InputAvailable())
      {
        char s[256];
        ReadInput(s, sizeof(s));
        if (Is(std::string(s), "exit", 2))
          break;
      }
    }
  }

  if (i == mvlist.Size())
  {
    int t1 = clock();
    double dt = (t1 - t0) / 1000.;

    out("\n Nodes: %d\n", sum);
    out(" Time:  %f\n", dt);
    out(" Knps:  %f\n\n", sum / dt / 1000.);
  }
}

const EVAL SEE_VALUE[14] =
{ 0, 0, 100, 100, 400, 400, 400, 400, 600, 600, 1200, 1200, 20000, 20000 };

EVAL Search::SEE_Exchange(const Position& pos, FLD to, COLOR side, EVAL currScore, EVAL target, U64 occ)
{
  U64 att = pos.GetAttacks(to, side, occ) & occ;
  if (att == 0)
    return currScore;

  FLD from = NF;
  PIECE piece;
  EVAL newTarget = SEE_VALUE[KW] + 1;

  while (att)
  {
    FLD f = Bitboard::PopLSB(att);
    piece = pos[f];
    if (SEE_VALUE[piece] < newTarget) 
    {
      from = f;
      newTarget = SEE_VALUE[piece];
    }
  }

  occ ^= Bitboard::Single(from);
  EVAL score = - SEE_Exchange(pos, to, side ^ 1, -(currScore + target), newTarget, occ);
  return (score > currScore)? score : currScore;
}

EVAL Search::SEE(const Position& pos, Move mv)
{
  FLD from = mv.From();
  FLD to = mv.To();
  PIECE piece = mv.Piece();
  PIECE captured = mv.Captured();
  PIECE promotion = mv.Promotion();
  COLOR side = GetColor(piece);

  EVAL score0 = SEE_VALUE[captured];
  if (promotion)
  {
    score0 += SEE_VALUE[promotion] - SEE_VALUE[PW];
    piece = promotion;
  }

  U64 occ = pos.BitsAll() ^ Bitboard::Single(from);
  EVAL score = - SEE_Exchange(pos, to, side ^ 1, -score0, SEE_VALUE[piece], occ);

  return score;
}

void Search::UpdateScores(MoveList& mvlist, Move hashmv, int ply)
{
  for (int i = 0; i < mvlist.Size(); ++i)
  {
    Move mv = mvlist[i]._mv;
    mvlist[i]._value = 0;

    if (mv == hashmv)
      mvlist[i]._value = SORT_HASH;
    else if (mv.Captured())
      mvlist[i]._value = SORT_CAPTURE + 10 * VALUE[mv.Captured()] - mv.Piece();
    else if (mv == _matekillers[ply])
      mvlist[i]._value = SORT_MATEKILLER;
    else if (mv == _killers[ply])
      mvlist[i]._value = SORT_KILLER;
    else
    {
      PIECE piece = mv.Piece();
      FLD to = mv.To();
      if (_histTry[piece][to])
        mvlist[i]._value = 100 * _histSuccess[piece][to] / _histTry[piece][to];
    }
  }
}

void Search::UpdateScoresQ(MoveList& mvlist)
{
  for (int i = 0; i < mvlist.Size(); ++i)
  {
    Move mv = mvlist[i]._mv;
    mvlist[i]._value = 0;

    if (mv.Captured())
      mvlist[i]._value = SORT_CAPTURE + 10 * VALUE[mv.Captured()] - mv.Piece();
    else
    {
      PIECE piece = mv.Piece();
      FLD to = mv.To();
      if (_histTry[piece][to])
        mvlist[i]._value = 100 * _histSuccess[piece][to] / _histTry[piece][to];
    }
  }
}
