//  GREKO Chess Engine
//  (c) 2002-2016 Vladimir Medvedev <vrm@bk.ru>
//  http://greko.su

//  search.h: chess tree search
//  modified: 30-Dec-2016

#ifndef SEARCH_H
#define SEARCH_H

#include <math.h>
#include "defaults.h"
#include "position.h"
#include "utils.h"

enum { MAX_PLY = 64 };
enum { MAX_BRANCH = 128 };
enum { IDLE, ANALYZE, THINKING, EPDTEST };

class Line
{
public:

	Line() : m_size(0) {}
	~Line() {}

	void Clear() { m_size = 0; }
	void Compose(Move mv, const Line& tail)
	{
		m_data[0] = mv;
		memcpy(m_data + 1, tail.m_data, tail.Size() * sizeof(Move));
		m_size = 1 + tail.Size();
	}
	size_t Size() const { return m_size; }
	Move operator[](size_t i) const { return m_data[i]; }

private:
	Move m_data[256];
	size_t m_size;
};

struct SearchParams
{
	SearchParams() :
		NullMoveReduction(DEFAULT_NULL_MOVE_REDUCTION),
		NullMoveMinDepth(DEFAULT_NULL_MOVE_MIN_DEPTH),
		PruningMargin1(DEFAULT_PRUNING_MARGIN_1),
		PruningMargin2(DEFAULT_PRUNING_MARGIN_2),
		PruningMargin3(DEFAULT_PRUNING_MARGIN_3),
		LmrMinDepth(DEFAULT_LMR_MIN_DEPTH),
		LmrMinMoveNumber(DEFAULT_LMR_MIN_MOVE_NUMBER),
		QChecks(DEFAULT_QCHECKS)
	{}

	int NullMoveReduction;
	int NullMoveMinDepth;
	int PruningMargin1;
	int PruningMargin2;
	int PruningMargin3;
	int LmrMinDepth;
	int LmrMinMoveNumber;
	int QChecks;
};

extern SearchParams g_searchParams;

enum
{
	HASH_UNKNOWN = 0x00,
	HASH_ALPHA   = 0x10,
	HASH_BETA    = 0x20,
	HASH_EXACT   = 0x40
};

struct HashEntry
{
	U64 m_hash;
	U32 m_mv;
	I16 m_eval;
	I8  m_depth;
	U8  m_flags;

	U8 Age() const { return m_flags & 0x0f; }
	U8 Type() const { return m_flags & 0xf0; }
};

struct MultiPVEntry
{
	Line m_pv;
	EVAL m_score;
	bool m_seen;
};

const int SORT_HASH       = 0x40000000;
const int SORT_CAPTURE    = 0x20000000;
const int SORT_MATEKILLER = 0x18000000;
const int SORT_KILLER     = 0x10000000;

void  Epdtest(FILE* psrc, double time_in_seconds, int reps);
void  SetHashMB(double mb);
void  SetKnps(double knps);
void  SetLimits(int restMillisec, int sd, NODES sn, int stHard, int stSoft);
void  SetMultiPV(int n);
void  StartAnalyze(const Position& pos);
void  StartThinking(Position& pos);
void  StartPerft(const Position& pos, int depth);

#endif
