//  GREKO Chess Engine
//  (c) 2002-2016 Vladimir Medvedev <vrm@bk.ru>
//  http://greko.su

//  search.cpp: chess tree search
//  modified: 30-Dec-2016

#include "book.h"
#include "eval.h"
#include "moves.h"
#include "notation.h"
#include "search.h"
#include "utils.h"

SearchParams g_searchParams;

extern Position g_pos;
extern std::list<std::string> g_commandQueue;
extern Book g_book;

EVAL  AlphaBetaRoot(EVAL alpha, EVAL beta, const int depth);
EVAL  AlphaBeta(EVAL alpha, EVAL beta, const int depth, int ply, bool isNull);
EVAL  AlphaBetaQ(EVAL alpha, EVAL beta, int ply, int qply);
void  CheckInput();
void  CheckLimits();
int   Extensions(Move mv, Move lastMove, bool check, int ply, bool onPV);
int   GetSearchTime();   // milliseconds
HashEntry* ProbeHash();
void  RecordHash(Move bestMove, int depth, EVAL eval, U8 type, int ply);
EVAL  SEE(const Position& pos, Move mv);
EVAL  SEE_Exchange(const Position& pos, FLD to, COLOR side, EVAL currScore, EVAL target, U64 occ);
void  SetLimits(int restMillisec, int sd, NODES sn, int stHard, int stSoft);
bool  StartEpd(const std::string& fen, int reps);
void  UpdateScores(MoveList& mvlist, Move hashmv, int ply);
void  UpdateScoresQ(MoveList& mvlist);

bool         g_flag = false;
U8           g_hashAge = 0;
HashEntry*   g_hash = NULL;
long         g_hashSize = 0;
int          g_histTry[14][64];
int          g_histSuccess[14][64];
int          g_iter;
EVAL         g_iterScore;
Move         g_killers[MAX_PLY];
double       g_knpsLimit = 999999;
MoveList     g_lists[MAX_PLY];
Move         g_matekillers[MAX_PLY];
int          g_multiPV = 1;
MultiPVEntry m_multiPVs[MAX_BRANCH];
int          g_mode = IDLE;
NODES        g_nodes;
Line         g_PV[MAX_PLY];
EVAL         g_rootAlpha;
EVAL         g_rootBeta;
Line         g_rootPV;
EVAL         g_rootWindow = VAL_P;
int          g_sd = 0;
Position     g_searchPos;
NODES        g_sn = 0;
U32          g_startTime;
int          g_stHard = 2000;
int          g_stSoft = 2000;

EVAL AlphaBetaRoot(EVAL alpha, EVAL beta, const int depth)
{
	g_rootAlpha = alpha;
	g_rootBeta = beta;

	int ply = 0;
	bool inCheck = g_searchPos.InCheck();
	g_PV[ply].Clear();
	g_nodes++;

	for (int j = 0; j < MAX_BRANCH; ++j)
	{
		m_multiPVs[j].m_score = -INFINITY_SCORE;
		m_multiPVs[j].m_pv.Clear();
	}

	Move hashMove = 0;
	U8 hashType = HASH_ALPHA;
	HashEntry* pentry = ProbeHash();
	if (pentry)
		hashMove = pentry->m_mv;

	MoveList& mvlist = g_lists[ply];
	if (inCheck)
		mvlist.GenCheckEvasions(g_searchPos);
	else
		mvlist.GenAllMoves(g_searchPos);
	UpdateScores(mvlist, hashMove, ply);

	EVAL e = 0;
	int legalMoves = 0;
	Move bestMove = 0;
	Move lastMove = g_searchPos.LastMove();

	for (int i = 0; i < mvlist.Size(); ++i)
	{
		Move mv = mvlist.GetNthBest(i, hashMove);
		if (g_searchPos.MakeMove(mv))
		{
			legalMoves++;
			g_histTry[mv.Piece()][mv.To()] += depth;

			if (g_protocol == UCI && GetSearchTime() > 1000)
			{
				out("info currmove %s", MoveToStrLong(mv));
				out(" currmovenumber %d\n", legalMoves);
			}

			int newDepth = depth - 1;
			newDepth += Extensions(mv, lastMove, inCheck, ply, true);

			if (g_multiPV > 1)
				e = -AlphaBeta(-beta - VAL_Q, -alpha + VAL_Q, newDepth, ply + 1, false);
			else
			{
				if (legalMoves == 1)
					e = -AlphaBeta(-beta, -alpha, newDepth, ply + 1, false);
				else
				{
					e = -AlphaBeta(-alpha - 1, -alpha, newDepth, ply + 1, false);
					if (e > alpha && e < beta)
						e = -AlphaBeta(-beta, -alpha, newDepth, ply + 1, false);
				}
			}
			g_searchPos.UnmakeMove();

			if (g_flag)
				return alpha;

			if (legalMoves == 1)
			{
				bestMove = mv;
				g_PV[ply].Compose(mv, g_PV[ply + 1]);
			}

			//
			//   Update multipv
			//

			if (legalMoves < MAX_BRANCH)
			{
				MultiPVEntry *mpv = &(m_multiPVs[legalMoves - 1]);
				mpv->m_score = e;
				mpv->m_pv.Compose(mv, g_PV[ply + 1]);
			}

			if (e > alpha)
			{
				bestMove = mv;
				if (!mv.Captured())
					g_histSuccess[mv.Piece()][mv.To()] += depth;
				hashType = HASH_EXACT;

				g_PV[ply].Compose(mv, g_PV[ply + 1]);
				alpha = e;

				if (alpha >= beta)
				{
					hashType = HASH_BETA;
					if (!mv.Captured())
					{
						if (e > CHECKMATE_SCORE - MAX_PLY && e <= CHECKMATE_SCORE)
							g_matekillers[ply] = mv;
						else
							g_killers[ply] = mv;
					}
					break;
				}
			}
		}
	}
	if (legalMoves == 0)
	{
		if (inCheck)
			alpha = -CHECKMATE_SCORE + ply;
		else
			alpha = DRAW_SCORE;
	}
	RecordHash(bestMove, depth, alpha, hashType, ply);
	return alpha;
}

EVAL AlphaBeta(EVAL alpha, EVAL beta, const int depth, int ply, bool isNull)
{
	g_PV[ply].Clear();
	++g_nodes;

	COLOR side = g_searchPos.Side();
	bool onPV = (beta - alpha > 1);
	bool inCheck = g_searchPos.InCheck();
	bool lateEndgame = g_searchPos.MatIndex(side) < 5;

	CheckLimits();
	if (g_nodes % 8192 == 0)
		CheckInput();

	if (g_flag)
		return alpha;

	//
	//   DRAW DETECTION
	//

	if (ply >= MAX_PLY)
		return DRAW_SCORE;

	EVAL_ESTIMATION ee = EstimateDraw(g_searchPos);
	if (ee == EVAL_THEORETICAL_DRAW || ee == EVAL_PRACTICAL_DRAW)
		return DRAW_SCORE;

	if (!isNull)
	{
		int rep_total = g_searchPos.GetRepetitions();
		if (rep_total >= 2) return DRAW_SCORE;
	}

	//
	//   PROBING HASH
	//

	Move hashMove = 0;
	U8 hashType = HASH_ALPHA;
	HashEntry* pentry = ProbeHash();

	if (pentry)
	{
		hashMove = pentry->m_mv;
		if (pentry->m_depth >= depth)
		{
			EVAL hash_eval = pentry->m_eval;
			if (hash_eval > CHECKMATE_SCORE - 50 && hash_eval <= CHECKMATE_SCORE)
				hash_eval -= ply;
			else if (hash_eval < -CHECKMATE_SCORE + 50 && hash_eval >= -CHECKMATE_SCORE)
				hash_eval += ply;

			if (pentry->Type() == HASH_EXACT)
				return hash_eval;
			else if (pentry->Type() == HASH_ALPHA && hash_eval <= alpha)
				return alpha;
			else if (pentry->Type() == HASH_BETA && hash_eval >= beta)
				return beta;
		}
	}

	//
	//   QSEARCH
	//

	if (depth <= 0 && !inCheck)
	{
		--g_nodes;
		return AlphaBetaQ(alpha, beta, ply, 0);
	}

	//
	//   PRUNING
	//

	EVAL MARGIN[4] =
	{
		INFINITY_SCORE,
		(EVAL)g_searchParams.PruningMargin1,
		(EVAL)g_searchParams.PruningMargin2,
		(EVAL)g_searchParams.PruningMargin3
	};

	if (!inCheck && !onPV && depth > 0 && depth < 4)
	{
		EVAL staticScore = Evaluate(g_searchPos, alpha - MARGIN[depth], beta + MARGIN[depth]);
		if (staticScore <= alpha - MARGIN[depth])
			return AlphaBetaQ(alpha, beta, ply, 0);
		if (staticScore >= beta + MARGIN[depth])
			return beta;
	}

	//
	//   NULL MOVE
	//

	const int R = g_searchParams.NullMoveReduction + depth / 6;
	if (!inCheck &&
		!onPV &&
		!isNull &&
		!lateEndgame &&
		depth >= g_searchParams.NullMoveMinDepth)
	{
		g_searchPos.MakeNullMove();
		EVAL nullEval;
		nullEval = -AlphaBeta(-beta, -beta + 1, depth - 1 - R, ply + 1, true);
		g_searchPos.UnmakeNullMove();
		if (nullEval >= beta)
			return beta;
	}

	//
	//   IID
	//

	if (onPV && hashMove == 0 && depth > 4)
	{
		AlphaBeta(alpha, beta, depth - 4, ply, isNull);
		if (g_PV[ply].Size() > 0)
			hashMove = g_PV[ply][0];
	}

	MoveList& mvlist = g_lists[ply];
	if (inCheck)
		mvlist.GenCheckEvasions(g_searchPos);
	else
		mvlist.GenAllMoves(g_searchPos);
	UpdateScores(mvlist, hashMove, ply);

	int legalMoves = 0, quietMoves = 0;
	EVAL e = 0;
	Move bestMove = 0;
	Move lastMove = g_searchPos.LastMove();

	for (int i = 0; i < mvlist.Size(); ++i)
	{
		Move mv = mvlist.GetNthBest(i, hashMove);
		if (g_searchPos.MakeMove(mv))
		{
			legalMoves++;
			g_histTry[mv.Piece()][mv.To()] += depth;
			int newDepth = depth - 1;
			bool givesCheck = g_searchPos.InCheck();
			newDepth += Extensions(mv, lastMove, inCheck, ply, onPV);

			//
			//   LMR
			//

			int reduction = 0;
			if (depth >= g_searchParams.LmrMinDepth &&
				!inCheck &&
				!onPV &&
				!givesCheck &&
				!mv.Captured() &&
				!mv.Promotion() &&
				!lateEndgame &&
				g_histSuccess[mv.Piece()][mv.To()] <= g_histTry[mv.Piece()][mv.To()] * 3 / 4 &&
				mv != g_killers[ply])
			{
				++quietMoves;
				if (quietMoves >= g_searchParams.LmrMinMoveNumber)
				{
					reduction = 1 + quietMoves / 6;
					newDepth -= reduction;
				}
			}

			if (legalMoves == 1)
				e = -AlphaBeta(-beta, -alpha, newDepth, ply + 1, false);
			else
			{
				e = -AlphaBeta(-alpha - 1, -alpha, newDepth, ply + 1, false);
				if (reduction > 0 && e > alpha)
				{
					newDepth += reduction;
					e = -AlphaBeta(-alpha - 1, -alpha, newDepth, ply + 1, false);
				}
				if (e > alpha && e < beta)
					e = -AlphaBeta(-beta, -alpha, newDepth, ply + 1, false);
			}
			g_searchPos.UnmakeMove();

			if (g_flag)
				return alpha;

			if (e > alpha)
			{
				bestMove = mv;
				if (!mv.Captured())
					g_histSuccess[mv.Piece()][mv.To()] += depth;
				hashType = HASH_EXACT;
				alpha = e;
				if (alpha >= beta)
				{
					hashType = HASH_BETA;
					if (!mv.Captured())
					{
						if (e > CHECKMATE_SCORE - MAX_PLY && e <= CHECKMATE_SCORE)
							g_matekillers[ply] = mv;
						else
							g_killers[ply] = mv;
					}
					break;
				}
				g_PV[ply].Compose(mv, g_PV[ply + 1]);
			}
		}
	}
	if (legalMoves == 0)
	{
		if (inCheck)
			alpha = -CHECKMATE_SCORE + ply;
		else
			alpha = DRAW_SCORE;
	}
	else if (g_searchPos.Fifty() >= 100)
		alpha = DRAW_SCORE;
	RecordHash(bestMove, depth, alpha, hashType, ply);
	return alpha;
}

EVAL AlphaBetaQ(EVAL alpha, EVAL beta, int ply, int qply)
{
	g_PV[ply].Clear();
	++g_nodes;

	CheckLimits();
	if (g_nodes % 8192 == 0)
		CheckInput();

	if (g_flag)
		return alpha;

	if (ply >= MAX_PLY) return DRAW_SCORE;

	bool inCheck = g_searchPos.InCheck();
	EVAL staticEval = Evaluate(g_searchPos, alpha, beta);

	if (!inCheck && staticEval > alpha)
		alpha = staticEval;

	if (alpha >= beta)
		return beta;

	MoveList& mvlist = g_lists[ply];
	if (inCheck)
		mvlist.GenCheckEvasions(g_searchPos);
	else
		mvlist.GenCaptures(g_searchPos, qply < g_searchParams.QChecks);
	UpdateScoresQ(mvlist);

	EVAL e = 0;
	int legalMoves = 0;
	for (int i = 0; i < mvlist.Size(); ++i)
	{
		Move mv = mvlist.GetNthBest(i);
		if (!inCheck && SEE(g_searchPos, mv) < 0)
			continue;

		if (g_searchPos.MakeMove(mv))
		{
			legalMoves++;
			e = -AlphaBetaQ(-beta, -alpha, ply + 1, qply + 1);
			g_searchPos.UnmakeMove();

			if (g_flag)
				return alpha;

			if (e > alpha)
			{
				alpha = e;
				if (alpha >= beta)
					break;
				g_PV[ply].Compose(mv, g_PV[ply + 1]);
			}
		}
	}
	if (inCheck && legalMoves == 0)
		alpha = -CHECKMATE_SCORE + ply;

	return alpha;
}

void CheckInput()
{
	if (g_rootPV.Size() == 0)
		return;

	if (InputAvailable())
	{
		char buf[4096];
		ReadInput(buf, sizeof(buf));
		TokenString ts(buf);
		std::string cmd = ts.GetToken();
		if (g_mode == ANALYZE)
		{
			Move mv = StrToMove(cmd, g_pos);
			if (mv)
			{
				g_flag = true;
				g_commandQueue.push_back("force");
				g_commandQueue.push_back(ts.Str());
				g_commandQueue.push_back("analyze");
			}
			else if (Is(cmd, "board", 1))
				g_pos.Print();
			else if (Is(cmd, "quit", 1))
				exit(0);
			else if (Is(cmd, "isready", 7))
				out("readyok\n");
			else if (Is(cmd, "exit", 2))
				g_flag = true;
			else if (Is(cmd, "stop", 2))
				g_flag = true;
			else if (Is(cmd, "new", 3))
				g_flag = true;
			else if (Is(cmd, "setoption", 8))
			{
				ts.GetToken(); // "name"
				std::string token = ts.GetToken(); // name
				if (token == "MultiPV")
				{
					ts.GetToken(); // "value"
					token = ts.GetToken(); // value
					g_multiPV = atoi(token.c_str());
				}
			}
			else if (Is(cmd, "undo", 4))
			{
				g_flag = true;
				g_commandQueue.push_back("undo");
				g_commandQueue.push_back("analyze");
			}
			else if (Is(cmd, "remove", 6))
			{
				g_flag = true;
				g_commandQueue.push_back("undo");
				g_commandQueue.push_back("analyze");
			}
		}
		else if (g_mode == THINKING)
		{
			if (Is(cmd, "quit", 1))
				exit(0);
			else if (Is(cmd, "isready", 7))
				out("readyok\n");
			else if (Is(cmd, "?", 1))
				g_flag = true;
			else if (Is(cmd, "stop", 4))
				g_flag = true;
			else if (Is(cmd, "new", 3))
				g_flag = true;
			else if (Is(cmd, "result", 6))
				g_flag = true;
			else if (Is(cmd, "setoption", 8))
			{
				ts.GetToken(); // "name"
				std::string token = ts.GetToken(); // name
				if (token == "MultiPV")
				{
					ts.GetToken(); // "value"
					token = ts.GetToken(); // value
					g_multiPV = atoi(token.c_str());
				}
			}
		}
		else if (g_mode == EPDTEST)
		{
			if (Is(cmd, "board", 1))
				g_pos.Print();
			else if (Is(cmd, "quit", 1))
				exit(0);
			else if (Is(cmd, "exit", 2))
				g_flag = true;
			else if (Is(cmd, "new", 3))
				g_flag = true;
		}
	}
}

void CheckLimits()
{
	if (g_rootPV.Size() == 0)
		return;

	if (g_stHard)
	{
		if (GetSearchTime() >= g_stHard)
		{
			if (g_mode == THINKING || g_mode == EPDTEST)
				g_flag = true;
		}
	}
	if (g_sn && g_nodes >= g_sn)
	{
		if (g_mode == THINKING || g_mode == EPDTEST)
			g_flag = true;
	}
}

void ClearHash()
{
	memset(g_hash, 0, g_hashSize * sizeof(HashEntry));
}

void ClearHistory()
{
	memset(g_histTry, 0, 64 * 14 * sizeof(g_histTry[0][0]));
	memset(g_histSuccess, 0, 64 * 14 * sizeof(g_histSuccess[0][0]));
}

void ClearKillers()
{
	memset(g_killers, 0, MAX_PLY * sizeof(Move));
	memset(g_matekillers, 0, MAX_PLY * sizeof(Move));
}

void Epdtest(FILE* src, double seconds, int reps)
{
	char fen[256];
	int dt = static_cast<int>(1000 * seconds);
	SetLimits(0, 0, 0, dt, dt);
	int total = 0, solved = 0;
	while (fgets(fen, sizeof(fen), src))
	{
		if (g_searchPos.SetFen(fen) == false) continue;
		out(fen);
		out("\n");
		++total;
		if (StartEpd(fen, reps)) ++solved;
		out("\nScore: %d / %d\n\n", solved, total);
	}
}

int Extensions(Move mv, Move lastMove, bool check, int ply, bool onPV)
{
	int ext = 0;
	if (check)
		++ext;
	else if (ply < g_iter + 10)
	{
		if (mv.Piece() == PW && (mv.To() / 8) == 1)
			++ext;
		else if (mv.Piece() == PB && (mv.To() / 8) == 6)
			++ext;
		else if (onPV && lastMove && mv.To() == lastMove.To() && lastMove.Captured())
			++ext;
	}
	return ext;
}

int GetSearchTime()
{
	return GetTime() - g_startTime;   // milliseconds
}

void LimitKnps()
{
	int searchTime = GetSearchTime();
	double knps = (searchTime > 0)? static_cast<double>(g_nodes) / searchTime : 0;
	if (!g_flag)
	{
		if (knps > g_knpsLimit)
		{
			int sleepTime = static_cast<int>(g_nodes / g_knpsLimit) - searchTime;
			if (g_stHard)
			{
				if (sleepTime > g_stHard - searchTime)
				{
					sleepTime = g_stHard - searchTime;
					g_flag = true;
				}
			}
			SleepMilliseconds(sleepTime);
		}
	}
}

NODES Perft(int depth)
{
	if (depth == 0)
		return 1;

	MoveList& mvlist = g_lists[depth];
	mvlist.GenAllMoves(g_searchPos);
	NODES sum = 0, delta = 0;
	for (int i = 0; i < mvlist.Size(); ++i)
	{
		Move mv = mvlist[i].m_mv;
		if (g_searchPos.MakeMove(mv))
		{
			delta = Perft(depth - 1);
			sum += delta;
			g_searchPos.UnmakeMove();
		}
	}
	return sum;
}

void PrintPV()
{
	Move mv = 0;

	if (g_protocol == UCI)
	{
		for (int j = 0; j < MAX_BRANCH; ++j)
			m_multiPVs[j].m_seen = false;

		for (int j = 1; j <= g_multiPV; ++j)
		{
			EVAL best_score = -INFINITY_SCORE;
			MultiPVEntry *best_mpv = NULL;
			MultiPVEntry *mpv = NULL;

			for (int k = 0; k < MAX_BRANCH; ++k)
			{
				mpv = &(m_multiPVs[k]);
				if (mpv->m_seen)
					continue;
				if (mpv->m_pv.Size() == 0)
					break;
				if (mpv->m_score > best_score)
				{
					best_score = mpv->m_score;
					best_mpv = mpv;
				}
			}

			if (best_mpv)
			{
				best_mpv->m_seen = true;
				mpv = best_mpv;
				out("info multipv %d depth %d score", j, g_iter);

				if (mpv->m_score > CHECKMATE_SCORE - 50)
					out(" mate %d", 1 + (CHECKMATE_SCORE - mpv->m_score) / 2);
				else if (mpv->m_score < -CHECKMATE_SCORE + 50)
					out(" mate -%d", (mpv->m_score + CHECKMATE_SCORE) / 2);
				else
					out(" cp %d", mpv->m_score);

				out(" time %ld", GetSearchTime());
				out(" nodes %ld", g_nodes);
				if (GetSearchTime() > 0)
				{
					double knps = static_cast<double>(g_nodes) / GetSearchTime();
					out (" nps %d", static_cast<int>(1000 * knps));
				}
				out(" pv ");
				for (size_t m = 0; m < mpv->m_pv.Size(); ++m)
				{
					mv = mpv->m_pv[m];
					out("%s ", MoveToStrLong(mv));
				}
				out("\n");
			}
		}
		if (GetSearchTime() > 0)
		{
			out("info time %ld", GetSearchTime());
			out(" nodes %ld", g_nodes);
			double knps = static_cast<double>(g_nodes) / GetSearchTime();
			out (" nps %d", static_cast<int>(1000 * knps));
		}
		out("\n");
		return;
	}
	else
	{
		out(" %2d", g_iter);
		out(" %9d ", g_iterScore);
		out(" %7d ", GetSearchTime() / 10);
		out(" %12d ", g_nodes);
		out(" ");
	}

	Position tmp = g_searchPos;
	int movenum = tmp.Ply() / 2 + 1;
	for (size_t m = 0; m < g_rootPV.Size(); ++m)
	{
		mv = g_rootPV[m];
		if (tmp.Side() == WHITE)
			out("%d. ", movenum++);
		else if (m == 0)
			out("%d. ... ", movenum++);

		MoveList mvlist;
		mvlist.GenAllMoves(tmp);
		out("%s", MoveToStrShort(mv, tmp));
		tmp.MakeMove(mv);

		if (tmp.InCheck())
		{
			if (int(g_iterScore + m + 1) == CHECKMATE_SCORE || int(g_iterScore - m - 1) == -CHECKMATE_SCORE)
			{
				out("#");
				if (g_iterScore > 0)
					out(" {+");
				else
					out(" {-");
				out("Mate in %d}", m / 2 + 1);
			}
			else
				out("+");
		}
		if (m == 0)
		{
			if (g_iterScore <= g_rootAlpha)
				out("?");
			else if (g_iterScore >= g_rootBeta)
				out("!");
		}
		out(" ");
	}
	out("\n");
}

HashEntry* ProbeHash()
{
	long index = long(g_searchPos.Hash() % g_hashSize);
	if (g_hash[index].m_hash == g_searchPos.Hash())
		return g_hash + index;
	else
		return NULL;
}

void RecordHash(Move bestMove, int depth, EVAL eval, U8 hashType, int ply)
{
	long index = long(g_searchPos.Hash() % g_hashSize);
	HashEntry* pentry = g_hash + index;

	if (depth > pentry->m_depth || g_hashAge != pentry->Age())
	{
		if (eval > CHECKMATE_SCORE - 50 && eval <= CHECKMATE_SCORE)
			eval += ply;
		else if (eval < -CHECKMATE_SCORE + 50 && eval >= -CHECKMATE_SCORE)
			eval -= ply;

		pentry->m_depth = static_cast<I8>(depth);
		pentry->m_eval = eval;
		pentry->m_hash = g_searchPos.Hash();
		pentry->m_mv = bestMove;
		pentry->m_flags = hashType | g_hashAge;
	}
}

void SetHashMB(double mb)
{
	if (g_hash)
		delete[] g_hash;

	g_hashSize = long (1024 * 1024 * mb / sizeof(HashEntry));
	if (g_hashSize <= 0)
		g_hashSize = 1;

	g_hash = new HashEntry[g_hashSize];

	if (g_protocol == CONSOLE)
	{
		out("main hash: %8ld nodes = ", g_hashSize);
		out("%.1lf MB\n", g_hashSize * sizeof(HashEntry) / (1024. * 1024.));
	}
}

void SetKnps(double knps)
{
	g_knpsLimit = knps;
}

void SetLimits(int restMillisec, int sd, NODES sn, int stHard, int stSoft)
{
	g_sd = sd;
	g_sn = sn;
	g_stHard = stHard;
	g_stSoft = stSoft;
}

void SetMultiPV(int n)
{
	g_multiPV = n;
}

void StartAnalyze(const Position& pos)
{
	g_startTime = GetTime();
	g_searchPos = pos;
	g_hashAge = (g_hashAge + 1) & 0x0f;
	g_mode = ANALYZE;
	g_flag = false;
	g_nodes = 0;

	SetLimits(0, 0, 0, 0, 0);

	ClearHash();
	ClearHistory();
	ClearKillers();

	if (g_protocol == CONSOLE)
		out("\n%s\n\n", g_searchPos.Fen());

	EVAL alpha = -INFINITY_SCORE;
	EVAL beta = INFINITY_SCORE;

	for (g_iter = 1; g_iter < MAX_PLY; g_iter++)
	{
		EVAL e = AlphaBetaRoot(alpha, beta, g_iter);
		LimitKnps();

		if (g_flag)
			break;

		g_iterScore = e;
		if (e > alpha)
			g_rootPV = g_PV[0];

		if (e > alpha && e < beta)
		{
			PrintPV();

			alpha = e - g_rootWindow / 2;
			beta = e + g_rootWindow / 2;
		}
		else
		{
			PrintPV();

			alpha = -INFINITY_SCORE;
			beta = INFINITY_SCORE;
			g_iter--;
		}
	}
}

bool StartEpd(const std::string& fen, int reps)
{
	g_hashAge = (g_hashAge + 1) & 0x0f;
	g_mode = EPDTEST;
	g_flag = false;
	g_nodes = 0;
	int counter = 0;

	ClearHash();
	ClearHistory();
	ClearKillers();

	g_startTime = GetTime();

	EVAL alpha = -INFINITY_SCORE;
	EVAL beta = INFINITY_SCORE;

	for (g_iter = 1; g_iter < MAX_PLY; g_iter++)
	{
		bool moveFound = false;
		EVAL e = AlphaBetaRoot(alpha, beta, g_iter);
		LimitKnps();

		if (g_flag)
			break;

		g_iterScore = e;
		if (e > alpha)
			g_rootPV = g_PV[0];

		if (g_rootPV.Size() > 0)
		{
			Move mv = g_rootPV[0];
			std::string mvstr = MoveToStrShort(mv, g_searchPos);

			if (fen.find(mvstr) != std::string::npos)
				moveFound = true;
			else
			{
				mvstr = MoveToStrLong(mv);
				if (fen.find(mvstr) != std::string::npos)
					moveFound = true;
			}

			if (fen.find("am") != std::string::npos && fen.find("bm") == std::string::npos)
				moveFound = !moveFound;
		}

		if (e > alpha && e < beta)
		{
			alpha = e - g_rootWindow / 2;
			beta = e + g_rootWindow / 2;
		}
		else
		{
			alpha = -INFINITY_SCORE;
			beta = INFINITY_SCORE;
			g_iter--;
		}

		if (moveFound)
		{
			++counter;
			Highlight(true);
			out(" yes ");
			Highlight(false);
		}
		else
		{
			counter = 0;
			out("  no ");
		}

		Highlight(counter >= reps);
		PrintPV();
		Highlight(false);

		if (counter >= reps)
			break;
	}

	return (counter > 0);
}

void StartThinking(Position& pos)
{
	g_startTime = GetTime();

	std::string message;
	if (pos.IsGameOver(message))
	{
		out("%s\n", message);
		return;
	}

	g_searchPos = pos;
	g_hashAge = (g_hashAge + 1) & 0x0f;
	g_rootPV.Clear();
	EVAL e = 0;

	g_mode = THINKING;
	g_flag = false;
	g_nodes = 0;

	// ClearHash();
	ClearHistory();
	ClearKillers();

	if (g_protocol == CONSOLE)
		out("\n%s\n\n", g_searchPos.Fen());

	EVAL alpha = -INFINITY_SCORE;
	EVAL beta = INFINITY_SCORE;
	Move best_move = 0;

	// Book

	std::string buf;
	Move book_move = g_book.GetMove(pos, buf);
	if (book_move)
	{
		best_move = book_move;
		if (g_protocol == CONSOLE || g_protocol == WINBOARD)
			out(" 0 0 0 0      (%s)", buf);
		g_flag = true;
		goto MAKE_MOVE;
	}

	for (g_iter = 1; g_iter < MAX_PLY; ++g_iter)
	{
		e = AlphaBetaRoot(alpha, beta, g_iter);
		LimitKnps();

		if (g_flag)
			break;

		g_iterScore = e;
		if (e > alpha)
			g_rootPV = g_PV[0];

		if (e > alpha && e < beta)
		{
			alpha = e - g_rootWindow / 2;
			beta = e + g_rootWindow / 2;

			if (g_stSoft)
			{
				if (GetSearchTime() >= g_stSoft)
					g_flag = true;
			}
			if (g_sd)
			{
				if (g_iter >= g_sd)
					g_flag = true;
			}
		}
		else
		{
			// Opening window
			if (e <= alpha)
				alpha = -INFINITY_SCORE;
			else
				beta = INFINITY_SCORE;
			--g_iter;
		}

		PrintPV();
		best_move = g_rootPV[0];

		if (best_move && (g_iter + e >= CHECKMATE_SCORE))
			g_flag = true;
	}

	if (g_iter == MAX_PLY && best_move)
		g_flag = true;

MAKE_MOVE:

	if (g_flag)
	{
		if (g_protocol == UCI)
			out("\nbestmove %s\n", MoveToStrLong(best_move));
		else
		{
			Highlight(true);
			out("\nmove %s\n", MoveToStrLong(best_move));
			Highlight(false);
			pos.MakeMove(best_move);
			if (pos.IsGameOver(message))
				out("%s\n", message);
			out("\n");
		}
	}
}

void StartPerft(const Position& pos, int depth)
{
	g_searchPos = pos;
	int t0 = GetTime();
	NODES sum = 0, delta = 0;

	MoveList& mvlist = g_lists[depth];
	mvlist.GenAllMoves(pos);

	out("\n");
	int i = 0;
	for (i = 0; i < mvlist.Size(); ++i)
	{
		Move mv = mvlist[i].m_mv;

		if (g_searchPos.MakeMove(mv))
		{
			delta = Perft(depth - 1);
			sum += delta;
			g_searchPos.UnmakeMove();

			out(" %s - ", MoveToStrLong(mv));
			out("%d\n", delta);
			if (InputAvailable())
			{
				char s[256];
				ReadInput(s, sizeof(s));
				if (Is(std::string(s), "exit", 2))
					break;
			}
		}
	}

	if (i == mvlist.Size())
	{
		int t1 = GetTime();
		double dt = (t1 - t0) / 1000.;

		out("\n Nodes: %d\n", sum);
		out(" Time:  %f\n", dt);
		out(" Knps:  %f\n\n", sum / dt / 1000.);
	}
}

const EVAL SEE_VALUE[14] =
{ 0, 0, 100, 100, 400, 400, 400, 400, 600, 600, 1200, 1200, 20000, 20000 };

EVAL SEE_Exchange(const Position& pos, FLD to, COLOR side, EVAL currScore, EVAL target, U64 occ)
{
	U64 att = pos.GetAttacks(to, side, occ) & occ;
	if (att == 0)
		return currScore;

	FLD from = NF;
	PIECE piece;
	EVAL newTarget = SEE_VALUE[KW] + 1;

	while (att)
	{
		FLD f = PopLSB(att);
		piece = pos[f];
		if (SEE_VALUE[piece] < newTarget)
		{
			from = f;
			newTarget = SEE_VALUE[piece];
		}
	}

	occ ^= BB_SINGLE[from];
	EVAL score = - SEE_Exchange(pos, to, side ^ 1, -(currScore + target), newTarget, occ);
	return (score > currScore)? score : currScore;
}

EVAL SEE(const Position& pos, Move mv)
{
	FLD from = mv.From();
	FLD to = mv.To();
	PIECE piece = mv.Piece();
	PIECE captured = mv.Captured();
	PIECE promotion = mv.Promotion();
	COLOR side = GetColor(piece);

	EVAL score0 = SEE_VALUE[captured];
	if (promotion)
	{
		score0 += SEE_VALUE[promotion] - SEE_VALUE[PW];
		piece = promotion;
	}

	U64 occ = pos.BitsAll() ^ BB_SINGLE[from];
	EVAL score = - SEE_Exchange(pos, to, side ^ 1, -score0, SEE_VALUE[piece], occ);

	return score;
}

void UpdateScores(MoveList& mvlist, Move hashmv, int ply)
{
	for (int i = 0; i < mvlist.Size(); ++i)
	{
		Move mv = mvlist[i].m_mv;
		mvlist[i].m_value = 0;

		if (mv == hashmv)
		{
			mvlist[i].m_value = SORT_HASH;
			std::swap(mvlist[0], mvlist[i]);
		}
		else if (mv.Captured())
			mvlist[i].m_value = SORT_CAPTURE + 10 * VALUE[mv.Captured()] - mv.Piece();
		else if (mv == g_matekillers[ply])
			mvlist[i].m_value = SORT_MATEKILLER;
		else if (mv == g_killers[ply])
			mvlist[i].m_value = SORT_KILLER;
		else
		{
			PIECE piece = mv.Piece();
			FLD to = mv.To();
			if (g_histTry[piece][to])
				mvlist[i].m_value = 100 * g_histSuccess[piece][to] / g_histTry[piece][to];
		}
	}
}

void UpdateScoresQ(MoveList& mvlist)
{
	for (int i = 0; i < mvlist.Size(); ++i)
	{
		Move mv = mvlist[i].m_mv;
		mvlist[i].m_value = 0;

		if (mv.Captured())
			mvlist[i].m_value = SORT_CAPTURE + 10 * VALUE[mv.Captured()] - mv.Piece();
		else
		{
			PIECE piece = mv.Piece();
			FLD to = mv.To();
			if (g_histTry[piece][to])
				mvlist[i].m_value = 100 * g_histSuccess[piece][to] / g_histTry[piece][to];
		}
	}
}

