//  GREKO Chess Engine
//  (c) 2002-2016 Vladimir Medvedev <vrm@bk.ru>
//  http://greko.su

//  main.cpp: initialize and start engine, command line interface
//  modified: 30-Dec-2016

#include "book.h"
#include "config.h"
#include "defaults.h"
#include "eval.h"
#include "learn.h"
#include "moves.h"
#include "notation.h"
#include "search.h"
#include "utils.h"

const std::string VERSION = "GreKo 2016";
const std::string RELEASE_DATE = "31-Dec-2016";

Book         g_book;
list<string> g_commandQueue;
bool         g_force = false;
int          g_inc = 0;
TokenString  g_line;
bool         g_run;
int          g_uciElo = 2600;
bool         g_uciLimitStrength = false;
Position     g_pos;
FILE*        g_log = NULL;

void SetTimeLimits(int restMillisec);

std::string GetToken()
{
	return g_line.GetToken();
}

void OnBk()
{
	std::string buf;
	g_book.GetMove(g_pos, buf);
	out("%s\n\n", buf);
}

void OnBook()
{
	std::string arg = GetToken();
	if (arg == "clean")
		g_book.Clean();
	else if (arg == "import")
	{
		std::string path = GetToken();
		std::string strMaxPly = GetToken();
		std::string strColor = GetToken();
		g_book.Import(path, strMaxPly, strColor);
	}
	else if (arg == "load")
	{
		std::string path = GetToken();
		g_book.Load(path);
	}
	else if (arg == "save")
	{
		std::string path = GetToken();
		g_book.Save(path);
	}
}

void OnConsole()
{
	g_protocol = CONSOLE;
}

void OnEpdtest()
{
	std::string strPath = GetToken();
	std::string strSec = GetToken();
	std::string strRep = GetToken();

	FILE *psrc = fopen(strPath.c_str(), "rt");
	if (!psrc)
	{
		out("Unable to open file '%s'\n", strPath);
		return;
	}
	double tm = strSec.empty()? 1.0 : atof(strSec.c_str());
	int reps = strRep.empty()? 3 : atoi(strRep.c_str());
	Epdtest(psrc, tm, reps);
	fclose(psrc);
}

void OnEval()
{
	EVAL e = Evaluate(g_pos, -INFINITY_SCORE, INFINITY_SCORE);
	out("eval = %d\n", e);
}

void OnFEN()
{
	out("%s\n\n", g_pos.Fen());
}

void OnGo()
{
	for (std::string token = GetToken(); token.length() > 0; token = GetToken())
	{
		if (token == "infinite")
			SetLimits(0, 0, 0, 0, 0);
		else if (token == "depth")
			SetLimits(0, atol(GetToken().c_str()), 0, 0, 0);
		else if (token == "nodes")
			SetLimits(0, 0, atol(GetToken().c_str()), 0, 0);
		else if (token == "movetime")
		{
			token = GetToken();
			SetLimits(atol(token.c_str()), 0, 0, atol(token.c_str()), atol(token.c_str()));
		}
		else if (token == "wtime" && g_pos.Side() == WHITE)
			SetTimeLimits(atoi(GetToken().c_str()));
		else if (token == "btime" && g_pos.Side() == BLACK)
			SetTimeLimits(atoi(GetToken().c_str()));
		else if(token == "winc" && g_pos.Side() == WHITE)
			g_inc = atoi(GetToken().c_str()); // in UCI time comes in milliseconds
		else if(token == "binc" && g_pos.Side() == BLACK)
			g_inc = atoi(GetToken().c_str()); // in UCI time comes in milliseconds
	}

	g_force = false;
	StartThinking(g_pos);
}

void OnLearn()
{
	std::string path = GetToken();
	if (path.empty())
	{
		out("Use: learn <file>\n");
		return;
	}

	int firstParam = 5;
	int lastParam = NUM_WEIGHTS - 1;
	std::string tk;

	tk = GetToken();
	if (!tk.empty()) firstParam = atoi(tk.c_str()) - 1;
	tk = GetToken();
	if (!tk.empty()) lastParam = atoi(tk.c_str()) - 1;

	if (firstParam < 0)
	{
		firstParam = 0;
		out("Incorrect firstParam, reset to %d\n", firstParam);
	}

	if (firstParam > NUM_WEIGHTS - 1)
	{
		firstParam = NUM_WEIGHTS - 1;
		out("Incorrect firstParam, reset to %d\n", firstParam);
	}

	if (lastParam < 0)
	{
		lastParam = 0;
		out("Incorrect lastParam, reset to %d\n", lastParam);
	}

	if (lastParam > NUM_WEIGHTS - 1)
	{
		lastParam = NUM_WEIGHTS - 1;
		out("Incorrect lastParam, reset to %d\n", lastParam);
	}

	std::string name = path;
	size_t ind1 = name.find(".pgn");
	if (ind1 != std::string::npos)
		name = name.substr(0, ind1);
	size_t ind2 = name.find(".fen");
	if (ind2 != std::string::npos)
		name = name.substr(0, ind2);

	std::string pgnFile = name + ".pgn";
	std::string fenFile = name + ".fen";

	FILE* fenSrc = fopen(fenFile.c_str(), "rt");
	if (fenSrc == NULL)
	{
		out("Creating file '%s'\n", fenFile.c_str());
		Pgn2Fen(pgnFile, fenFile);
	}
	else
	{
		out("Found file '%s'\n", fenFile.c_str());
		fclose(fenSrc);
	}

	Learn(fenFile, firstParam, lastParam);
}

void OnLevel()
{
	GetToken(); // mps
	GetToken(); // base
	int inc = atoi(GetToken().c_str()); // inc
	g_inc = 1000 * inc; // in WB increment comes in seconds
}

void OnList()
{
	MoveList mvlist;
	mvlist.GenAllMoves(g_pos);
	for (int i = 0; i < mvlist.Size(); ++i)
	{
		out("%s ", MoveToStrLong(mvlist[i].m_mv));
	}
	out(" -- total %d moves\n", mvlist.Size());
}

void OnLoad()
{
	std::string path = GetToken();
	FILE* psrc = fopen(path.c_str(), "rt");
	if (psrc == NULL)
	{
		out("%s not found\n", path);
		return;
	}
	std::string tmp = GetToken();
	int lineNum = tmp.empty()? 1 : atoi(tmp.c_str());
	char buf[256];
	for (int i = 0; i < lineNum; ++i)
	{
		if (fgets(buf, sizeof(buf), psrc) != NULL)
		{
			if (buf[strlen(buf) - 1] == '\n' || buf[strlen(buf) - 1] == '\r') buf[strlen(buf) - 1] = 0;
		}
	}
	if (g_pos.SetFen(buf))
		out("%s\n\n", buf);
	else
		out("Incorrect line number\n\n");
	fclose(psrc);
}

void OnMT()
{
	std::string path = GetToken();
	FILE* psrc = fopen(path.c_str(), "rt");
	if (psrc == NULL)
	{
		out("%s not found\n", path);
		return;
	}

	Position tmp = g_pos;
	char buf[256];
	while (fgets(buf, sizeof(buf), psrc))
	{
		if (buf[strlen(buf) - 1] == '\n' || buf[strlen(buf) - 1] == '\r') buf[strlen(buf) - 1] = 0;
		out("%s\n", buf);
		g_pos.SetFen(buf);

		EVAL e1 = Evaluate(g_pos, -INFINITY_SCORE, INFINITY_SCORE);
		g_pos.Mirror();
		EVAL e2 = Evaluate(g_pos, -INFINITY_SCORE, INFINITY_SCORE);

		if (e1 != e2)
		{
			out("Incorrect evaluation:\n");
			out("e1 = %d\n", e1);
			out("e2 = %d\n", e2);
			break;
		}
	}
	fclose(psrc);
	g_pos = tmp;
	out("\n");
}

void OnNew()
{
	g_force = false;
	g_pos.SetInitial();
}

void OnPerft()
{
	std::string token = GetToken();
	StartPerft(g_pos, atoi(token.c_str()));
}

void OnPosition()
{
	for (std::string token = GetToken(); token.length() > 0; token = GetToken())
	{
		if (token == "startpos")
		{
			g_pos.SetInitial();
		}
		else if (token == "moves")
		{
			for (token = GetToken(); token.length() > 0; token = GetToken())
			{
				Move mv = StrToMove(token, g_pos);
				if (mv)
					g_pos.MakeMove(mv);
				else
					break;
			}
		}
		else if (token == "fen")
		{
			std::string fen;
			for (token = GetToken(); token.length() > 0; token = GetToken())
			{
				if (token == "moves") break;
				fen += token;
				fen += " ";
			}
			g_pos.SetFen(fen);
			for (token = GetToken(); token.length() > 0; token = GetToken())
			{
				Move mv = StrToMove(token, g_pos);
				if (mv)
					g_pos.MakeMove(mv);
				else
					break;
			}
		}
	}
}

void OnProtover()
{
	out("\nfeature myname=\"%s\"", VERSION);
	out(" setboard=1 analyze=1 colors=0 san=0 ping=1 name=1 done=1\n\n");
}

void OnRemove()
{
	g_pos.UnmakeMove();
	g_pos.UnmakeMove();
}

void OnUCI()
{
	g_protocol = UCI;
	out("id name %s\n", VERSION);
	out("id author Vladimir Medvedev\n");
	out("option name Hash type spin default %d min 1 max 1024\n", DEFAULT_HASH_MB);
	out("option name MultiPV type spin default %d min 1 max 64\n", DEFAULT_MULTI_PV);
	out("option name UCI_LimitStrength type check default %s\n", DEFAULT_UCI_LIMIT_STRENGTH);
	out("option name UCI_Elo type spin default %d min 1600 max 2400\n", DEFAULT_UCI_ELO);
	out("option name LimitKnps type spin default %d min 1 max 9999\n", DEFAULT_LIMIT_KNPS);
	out("option name NullMoveReduction type spin default %d min 1 max 5\n", DEFAULT_NULL_MOVE_REDUCTION);
	out("option name NullMoveMinDepth type spin default %d min 1 max 9999\n", DEFAULT_NULL_MOVE_MIN_DEPTH);
	out("option name PruningMargin1 type spin default %d min 1 max 9999\n", DEFAULT_PRUNING_MARGIN_1);
	out("option name PruningMargin2 type spin default %d min 1 max 9999\n", DEFAULT_PRUNING_MARGIN_2);
	out("option name PruningMargin3 type spin default %d min 1 max 9999\n", DEFAULT_PRUNING_MARGIN_3);
	out("option name LmrMinDepth type spin default %d min 1 max 9999\n", DEFAULT_LMR_MIN_DEPTH);
	out("option name LmrMinMoveNumber type spin default %d min 1 max 9999\n", DEFAULT_LMR_MIN_MOVE_NUMBER);
	out("option name QChecks type spin default %d min 0 max 64\n", DEFAULT_QCHECKS);
	out("option name PositionalKnowledge type spin default %d min 0 max 200\n", DEFAULT_POSITIONAL_KNOWLEDGE);
	out("uciok\n");
}

void OnSD()
{
	int sd = atoi(GetToken().c_str());
	SetLimits(0, sd, 0, 0, 0);
}

void OnSetboard()
{
	std::string fen;
	std::string token;
	while ((token = GetToken()) != "")
	{
		fen += token;
		fen += " ";
	}
	g_pos.SetFen(fen);
}

void OnSetoption()
{
	GetToken(); // "name"
	std::string key = GetToken();
	GetToken(); // value
	std::string value = GetToken();

	if (CaseInsensitiveEquals(key, "MultiPV"))
		SetMultiPV(atoi(value.c_str()));
	else if (CaseInsensitiveEquals(key, "Hash"))
		SetHashMB(atof(value.c_str()));
	else if (CaseInsensitiveEquals(key, "UCI_LimitStrength"))
		g_uciLimitStrength = (value != "false" && value != "0");
	else if (CaseInsensitiveEquals(key, "UCI_Elo"))
	{
		if (g_uciLimitStrength)
		{
			g_uciElo = atoi(value.c_str());
			// double knps = pow(10, (m_uciElo - 1600) / 400.);
			double knps = 0.01 + (pow(2, (g_uciElo - 1350) / 70.) / 200);  // formula suggested by Alexander Schmidt
			out("knps=%f\n", knps);
			SetKnps(knps);
		}
	}
	else if (CaseInsensitiveEquals(key, "LimitKnps"))
	{
		if (!g_uciLimitStrength)
			SetKnps(atof(value.c_str()));
	}
	else if (CaseInsensitiveEquals(key, "NullMoveReduction"))
		g_searchParams.NullMoveReduction = atoi(value.c_str());
	else if (CaseInsensitiveEquals(key, "NullMoveMinDepth"))
		g_searchParams.NullMoveMinDepth = atoi(value.c_str());
	else if (CaseInsensitiveEquals(key, "PruningMargin1"))
		g_searchParams.PruningMargin1 = atoi(value.c_str());
	else if (CaseInsensitiveEquals(key, "PruningMargin2"))
		g_searchParams.PruningMargin2 = atoi(value.c_str());
	else if (CaseInsensitiveEquals(key, "PruningMargin3"))
		g_searchParams.PruningMargin3 = atoi(value.c_str());
	else if (CaseInsensitiveEquals(key, "LmrMinDepth"))
		g_searchParams.LmrMinDepth = atoi(value.c_str());
	else if (CaseInsensitiveEquals(key, "LmrMinMoveNumber"))
		g_searchParams.LmrMinMoveNumber = atoi(value.c_str());
	else if (CaseInsensitiveEquals(key, "QChecks"))
		g_searchParams.QChecks = atoi(value.c_str());
}

void OnSN()
{
	NODES sn = atol(GetToken().c_str());
	SetLimits(0, 0, sn, 0, 0);
}

void OnST()
{
	int dt = int(1000 * atof(GetToken().c_str()));
	SetLimits(dt, 0, 0, dt, dt);
}

void OnTime()
{
	SetTimeLimits(10 * atoi(GetToken().c_str())); // in WB time comes in centi-seconds
}

void RunCommandLine()
{
	out("\n");
	g_run = true;
	while (g_run)
	{
		char buf[4096];
		if (g_commandQueue.empty())
		{
			if (g_protocol == CONSOLE)
			{
				if (g_pos.Side() == WHITE)
					out("White(%d): ", g_pos.Ply() / 2 + 1);
				else
					out("Black(%d): ", g_pos.Ply() / 2 + 1);
			}
			ReadInput(buf, sizeof(buf));
			g_line = TokenString(buf);
		}
		else
		{
			g_line = TokenString(g_commandQueue.front());
			g_commandQueue.pop_front();
		}

		std::string cmd = GetToken();
		if (cmd.empty()) continue;

		Move mv = StrToMove(cmd, g_pos);
		if (mv)
		{
			g_pos.MakeMove(mv);
			if (!g_force)
				StartThinking(g_pos);
			continue;
		}

#define ON_CMD(pattern, minLen, action)    \
        if (Is(cmd, #pattern, minLen)) \
        {                              \
            action;                    \
            continue;                  \
        }

		ON_CMD (analyze,   2, StartAnalyze(g_pos))
		ON_CMD (bk,        2, OnBk())
		ON_CMD (board,     1, g_pos.Print())
		ON_CMD (book,      4, OnBook())
		ON_CMD (console,   3, OnConsole())
		ON_CMD (epdtest,   3, OnEpdtest())
		ON_CMD (eval,      1, OnEval())
		ON_CMD (fen,       3, OnFEN())
		ON_CMD (force,     5, g_force = true)
		ON_CMD (go,        2, OnGo())
		ON_CMD (isready,   7, out("readyok\n"))
		ON_CMD (learn,     3, OnLearn())
		ON_CMD (level,     3, OnLevel())
		ON_CMD (list,      2, OnList())
		ON_CMD (load,      2, OnLoad())
		ON_CMD (mirror,    2, g_pos.Mirror())
		ON_CMD (mt,        2, OnMT())
		ON_CMD (new,       3, OnNew())
		ON_CMD (perft,     3, OnPerft())
		ON_CMD (ping,      4, out("pong %s\n", GetToken()))
		ON_CMD (position,  8, OnPosition())
		ON_CMD (protover,  8, OnProtover())
		ON_CMD (quit,      1, g_run = false)
		ON_CMD (remove,    6, OnRemove())
		ON_CMD (sd,        2, OnSD())
		ON_CMD (setboard,  8, OnSetboard())
		ON_CMD (setoption, 8, OnSetoption())
		ON_CMD (sn,        2, OnSN())
		ON_CMD (st,        2, OnST())
		ON_CMD (time,      2, OnTime())
		ON_CMD (xboard,    6, g_protocol = WINBOARD)
		ON_CMD (uci,       3, OnUCI())
		ON_CMD (undo,      1, g_pos.UnmakeMove())
		if (g_protocol == CONSOLE)
			out("Unknown command: %s\n", g_line.Str());
	}
}

void SetTimeLimits(int restMillisec)
{
	int stHard = restMillisec / 2;
	int stSoft = restMillisec / 60 + g_inc;
	SetLimits(restMillisec, 0, 0, stHard, stSoft);
}

void Init()
{
	Config config("GreKo.ini");
	if (config.GetInt("WriteLog", 0)) g_log = fopen("greko.log", "at");

	InitBitboards();
	Position::InitHashNumbers();

	SetHashMB(config.GetDouble("HashMB", DEFAULT_HASH_MB));
	SetKnps(config.GetDouble("LimitKnps", DEFAULT_LIMIT_KNPS));

	g_searchParams.NullMoveReduction = config.GetInt("NullMoveReduction", DEFAULT_NULL_MOVE_REDUCTION);
	g_searchParams.NullMoveMinDepth = config.GetInt("NullMoveMinDepth", DEFAULT_NULL_MOVE_MIN_DEPTH);

	g_searchParams.PruningMargin1 = config.GetInt("PruningMargin1", DEFAULT_PRUNING_MARGIN_1);
	g_searchParams.PruningMargin2 = config.GetInt("PruningMargin2", DEFAULT_PRUNING_MARGIN_2);
	g_searchParams.PruningMargin3 = config.GetInt("PruningMargin3", DEFAULT_PRUNING_MARGIN_3);

	g_searchParams.LmrMinDepth = config.GetInt("LmrMinDepth", DEFAULT_LMR_MIN_DEPTH);
	g_searchParams.LmrMinMoveNumber = config.GetInt("LmrMinMoveNumber", DEFAULT_LMR_MIN_MOVE_NUMBER);

	g_searchParams.QChecks = config.GetInt("QChecks", DEFAULT_QCHECKS);
	g_positionalKnowledge = config.GetInt("PositionalKnowledge", DEFAULT_POSITIONAL_KNOWLEDGE);

	g_book.Init();
	RandSeed32(U32(time(0)));

	g_pos.SetInitial();
	InitEval();
}

int main()
{
	InitInput();

	Highlight(true);
	out("\n%s (%s)\n\n", VERSION.c_str(), RELEASE_DATE.c_str());
	Highlight(false);

	Init();
	RunCommandLine();
	return 0;
}

