//  GREKO Chess Engine
//  (c) 2002-2016 Vladimir Medvedev <vrm@bk.ru>
//  http://greko.su

//  eval.cpp: static position evaluation
//  modified: 30-Dec-2016

#include "eval.h"
#include "utils.h"

EVAL VAL_P = 100;                  // 1
EVAL VAL_N = 400;                  // 2
EVAL VAL_B = 400;                  // 3
EVAL VAL_R = 600;                  // 4
EVAL VAL_Q = 1200;                 // 5

EVAL VALUE[14];

EVAL PawnEndgameBonus     = 5;     // 6
EVAL PawnDoubled          = -10;   // 7
EVAL PawnIsolated         = -15;   // 8
EVAL PawnCenter           =  10;   // 9
EVAL PawnPassed           = 130;   //10
EVAL PawnPassedKingDist   = 10;    // 11
EVAL PawnPassedSquare     = 200;   // 12
EVAL PawnPassedBlocker    = 15;    // 13
EVAL KnightCenter         = 35;    // 14
EVAL KnightKingTropism    = 15;    // 15
EVAL BishopPairEndgame    = 60;    // 16
EVAL BishopMobility       = 50;    // 17
EVAL BishopKingTropism    = 10;    // 18
EVAL Rook7th              = 30;    // 19
EVAL RookSemiopen         = 20;    // 20
EVAL RookOpen             = 25;    // 21
EVAL RookMobility         = 25;    // 22
EVAL RookKingTropism      = 15;    // 23
EVAL QueenKingTropism     = 80;    // 24
EVAL QueenCenterEndgame   = 20;    // 25
EVAL Queen7th             = 20;    // 26
EVAL KingCenterMidgame    = -35;   // 27
EVAL KingCenterEndgame    = 30;    // 28
EVAL KingPawnShield       = 120;   // 29
EVAL AttackStrongerPiece  = 40;    // 30
EVAL WeakField            = -10;   // 31
EVAL MinorOnStrongField   = 10;    // 32

int g_positionalKnowledge = DEFAULT_POSITIONAL_KNOWLEDGE;

std::string paramNames[NUM_WEIGHTS] =
{
	"VAL_P",
	"VAL_N",
	"VAL_B",
	"VAL_R",
	"VAL_Q",
	"PawnEndgameBonus",
	"PawnDoubled",
	"PawnIsolated",
	"PawnCenter",
	"PawnPassed",
	"PawnPassedKingDist",
	"PawnPassedSquare",
	"PawnPassedBlocker",
	"KnightCenter",
	"KnightKingTropism",
	"BishopPairEndgame",
	"BishopMobility",
	"BishopKingTropism",
	"Rook7th",
	"RookSemiopen",
	"RookOpen",
	"RookMobility",
	"RookKingTropism",
	"QueenKingTropism",
	"QueenCenterEndgame",
	"Queen7th",
	"KingCenterMidgame",
	"KingCenterEndgame",
	"KingPawnShield",
	"AttackStrongerPiece",
	"WeakField",
	"MinorOnStrongField"
};

int x_min[NUM_WEIGHTS] =
{
	0,      // VAL_P
	0,      // VAL_N
	0,      // VAL_B
	0,      // VAL_R
	0,      // VAL_Q
	0,      // PawnEndgameBonus
	-200,   // PawnDoubled
	-200,   // PawnIsolated
	0,      // PawnCenter
	0,      // PawnPassed
	0,      // PawnPassedBlocker
	0,      // PawnPassedKingDist
	0,      // PawnPassedSquare
	0,      // KnightCenter
	0,      // KnightKingTropism
	0,      // BishopPairEndgame
	0,      // BishopMobility
	0,      // BishopKingTropism
	0,      // Rook7th
	0,      // RookSemiopen
	0,      // RookOpen
	0,      // RookMobility
	0,      // RookKingTropism
	0,      // QueenKingTropism
	0,      // QueenCenterEndgame
	0,      // Queen7th
	-200,   // KingCenterMidgame
	0,      // KingCenterEndgame
	0,      // KingPawnShield
	0,      // AttackStrongerPiece
	-200,   // WeakField
	0       // MinorOnStrongField
};

int x_max[NUM_WEIGHTS] =
{
	2000,  // VAL_P
	2000,  // VAL_N
	2000,  // VAL_B
	2000,  // VAL_R
	2000,  // VAL_Q
	200,     // PawnEndgameBonus
	0,     // PawnDoubled
	0,     // PawnIsolated
	200,   // PawnCenter
	200,   // PawnPassed
	200,   // PawnPassedBlocker
	200,   // PawnPassedKingDist
	200,   // PawnPassedSquare
	200,   // KnightCenter
	200,   // KnightKingTropism
	200,   // BishopPairEndgame
	200,   // BishopMobility
	200,   // BishopKingTropism
	200,   // Rook7th
	200,   // RookSemiopen
	200,   // RookOpen
	200,   // RookMobility
	200,   // RookKingTropism
	200,   // QueenKingTropism
	200,   // QueenCenterEndgame
	200,   // Queen7th
	0,     // KingCenterMidgame
	200,   // KingCenterEndgame
	200,   // KingPawnShield
	200,   // AttackStrongerPiece
	0,     // WeakField
	200    // MinorOnStrongField
};

std::string ParamName(int index)
{
	if (index < 0 || index >= NUM_WEIGHTS)
		return "(incorrect index)";
	else
		return paramNames[index];
}

EVAL PSQ_P[64];
EVAL PSQ_N[64];
EVAL PSQ_Q_END[64];
EVAL PSQ_K_MID[64];
EVAL PSQ_K_END[64];

EVAL PAWN_PASSED[8];
EVAL KNIGHT_KING_TROPISM[10];
EVAL BISHOP_MOBILITY[14];
EVAL BISHOP_KING_TROPISM[10];
EVAL ROOK_MOBILITY[15];
EVAL ROOK_KING_TROPISM[10];
EVAL QUEEN_KING_TROPISM[10];
EVAL KING_PAWN_SHIELD[10];

double InterpolateSquare(double arg, double argFrom, double argTo, double valFrom, double valTo)
{
	// A*x*x + B*x + C
	double x = arg - argFrom;
	double xMax = argTo - argFrom;
	double A = (valTo - valFrom) / (xMax * xMax);
	double B = 0;
	double C = valFrom;
	return A * x * x + B * x + C;
}

double InterpolateLinear(double arg, double argFrom, double argTo, double valFrom, double valTo)
{
	// A*x + B
	return valFrom + (valTo - valFrom) * (arg - argFrom) / (argTo - argFrom);
}

struct PawnEntry
{
	U32  m_pawnHash;
	int  m_ranks[10][2];
	EVAL m_score;
	U64  m_passed[2];
	U64  m_attacks[2];
	U64  m_potentialAttacks[2];

	void Read(const Position& pos);
};

const int g_pawnHashSize = 8192;
PawnEntry g_pawnHash[g_pawnHashSize];

inline int Dist(FLD f1, FLD f2)
{
	static const int dist[100] =
	{
		0, 1, 1, 1, 2, 2, 2, 2, 2, 3,
		3, 3, 3, 3, 3, 3, 4, 4, 4, 4,
		4, 4, 4, 4, 4, 5, 5, 5, 5, 5,
		5, 5, 5, 5, 5, 5, 6, 6, 6, 6,
		6, 6, 6, 6, 6, 6, 6, 6, 6, 7,
		7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
		7, 7, 7, 7, 8, 8, 8, 8, 8, 8,
		8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
		8, 9, 9, 9, 9, 9, 9, 9, 9, 9,
		9, 9, 9, 9, 9, 9, 9, 9, 9, 9
	};

	int drow = Row(f1) - Row(f2);
	int dcol = Col(f1) - Col(f2);
	int d2 = drow * drow + dcol * dcol;

	return dist[d2];
}

int PawnShieldWhite(const PawnEntry& pentry, FLD K)
{
	int r = 0;
	int file = Col(K) + 1;
	for (int i = file - 1; i <= file + 1; ++i)
	{
		int rank = pentry.m_ranks[i][WHITE];
		if (rank == 6)
			;
		else if (rank == 5)
			r += 1;
		else if (rank == 4)
			r += 2;
		else
			r += 3;
	}
	return r;
}

int PawnShieldBlack(const PawnEntry& pentry, FLD K)
{
	int r = 0;
	int file = Col(K) + 1;
	for (int i = file - 1; i <= file + 1; ++i)
	{
		int rank = pentry.m_ranks[i][BLACK];
		if (rank == 1)
			;
		else if (rank == 2)
			r += 1;
		else if (rank == 3)
			r += 2;
		else
			r += 3;
	}
	return r;
}

EVAL Evaluate(const Position& pos, EVAL alpha, EVAL beta)
{
	EVAL_ESTIMATION ee = EstimateDraw(pos);
	if (ee == EVAL_THEORETICAL_DRAW || ee == EVAL_PRACTICAL_DRAW)
		return DRAW_SCORE;

	EVAL matScore = pos.Material(WHITE) - pos.Material(BLACK);
	if (pos.Count(BW) == 2)
		matScore += BishopPairEndgame * (32 - pos.MatIndex(BLACK)) / 32;
	if (pos.Count(BB) == 2)
		matScore -= BishopPairEndgame * (32 - pos.MatIndex(WHITE)) / 32;

	matScore += pos.Count(PW) * PawnEndgameBonus * (32 - pos.MatIndex(BLACK)) / 32;
	matScore -= pos.Count(PB) * PawnEndgameBonus * (32 - pos.MatIndex(WHITE)) / 32;

	EVAL lazy = (pos.Side() == WHITE)? matScore : -matScore;
	if (lazy <= alpha - 3 * VAL_P / 2)
		return alpha;
	if (lazy >= beta + 3 * VAL_P / 2)
		return beta;

	EVAL posScore = 0;

	U64 x, y, occ = pos.BitsAll();
	FLD f;

	//
	//   PAWNS
	//

	int index = pos.PawnHash() % g_pawnHashSize;

	PawnEntry& pentry = g_pawnHash[index];
	if (pentry.m_pawnHash != pos.PawnHash())
		pentry.Read(pos);

	posScore += pentry.m_score;

	y = UpRight(pos.Bits(PW)) | UpLeft(pos.Bits(PW));
	posScore += AttackStrongerPiece * CountBits(y & (pos.Bits(NB) | pos.Bits(BB) | pos.Bits(RB) | pos.Bits(QB)));

	y = DownRight(pos.Bits(PB)) | DownLeft(pos.Bits(PB));
	posScore -= AttackStrongerPiece * CountBits(y & (pos.Bits(NW) | pos.Bits(BW) | pos.Bits(RW) | pos.Bits(QW)));

	x = pentry.m_passed[WHITE];
	while (x)
	{
		f = PopLSB(x);
		int adv = 7 - Row(f);
		posScore += PAWN_PASSED[adv];

		PIECE blocker = pos[f - 8];
		if (blocker != NOPIECE && GetColor(blocker) == BLACK)
			posScore -= PawnPassedBlocker;

		if (pos.MatIndex(BLACK) == 0)
		{
			FLD f1 = f;
			if (pos.Side() == BLACK)
				f1 += 8;
			if ((BB_PAWN_SQUARE[f1][WHITE] & pos.Bits(KB)) == 0)
				posScore += PawnPassedSquare * (7 - Row(f1)) / 6;
		}
		else if (pos.MatIndex(BLACK) < 10)
			posScore += PawnPassedKingDist * Dist(f - 8, pos.King(BLACK));
	}

	x = pentry.m_passed[BLACK];
	while (x)
	{
		f = PopLSB(x);
		int adv = Row(f);
		posScore -= PAWN_PASSED[adv];

		PIECE blocker = pos[f + 8];
		if (blocker != NOPIECE && GetColor(blocker) == WHITE)
			posScore += PawnPassedBlocker;

		if (pos.MatIndex(WHITE) == 0)
		{
			FLD f1 = f;
			if (pos.Side() == WHITE)
				f1 -= 8;
			if ((BB_PAWN_SQUARE[f1][BLACK] & pos.Bits(KW)) == 0)
				posScore -= PawnPassedSquare * Row(f1) / 6;
		}
		else if (pos.MatIndex(WHITE) < 10)
			posScore -= PawnPassedKingDist * Dist(f + 8, pos.King(WHITE));
	}

	//
	//   KNIGHTS
	//

	x = pos.Bits(NW);
	while (x)
	{
		f = PopLSB(x);
		posScore += KNIGHT_KING_TROPISM[Dist(f, pos.King(BLACK))];
		posScore += PSQ_N[f];
		y = BB_KNIGHT_ATTACKS[f];
		posScore += AttackStrongerPiece * CountBits(y & (pos.Bits(RB) | pos.Bits(QB)));
	}

	x = pos.Bits(NB);
	while (x)
	{
		f = PopLSB(x);
		posScore -= KNIGHT_KING_TROPISM[Dist(f, pos.King(WHITE))];
		posScore -= PSQ_N[FLIP[f]];
		y = BB_KNIGHT_ATTACKS[f];
		posScore -= AttackStrongerPiece * CountBits(y & (pos.Bits(RW) | pos.Bits(QW)));
	}

	//
	//   BISHOPS
	//

	x = pos.Bits(BW);
	while (x)
	{
		f = PopLSB(x);
		posScore += BISHOP_KING_TROPISM[Dist(f, pos.King(BLACK))];
		y = BishopAttacks(f, occ);
		posScore += BISHOP_MOBILITY[CountBits(y)];
		posScore += AttackStrongerPiece * CountBits(y & (pos.Bits(RB) | pos.Bits(QB)));
	}

	x = pos.Bits(BB);
	while (x)
	{
		f = PopLSB(x);
		posScore -= BISHOP_KING_TROPISM[Dist(f, pos.King(WHITE))];
		y = BishopAttacks(f, occ);
		posScore -= BISHOP_MOBILITY[CountBits(y)];
		posScore -= AttackStrongerPiece * CountBits(y & (pos.Bits(RW) | pos.Bits(QW)));
	}

	//
	//   ROOKS
	//

	x = pos.Bits(RW);
	while (x)
	{
		f = PopLSB(x);
		posScore += ROOK_KING_TROPISM[Dist(f, pos.King(BLACK))];
		y = RookAttacks(f, occ ^ pos.Bits(RW));
		posScore += ROOK_MOBILITY[CountBits(y)];
		posScore += AttackStrongerPiece * CountBits(y & pos.Bits(QB));
		if (Row(f) == 1) posScore += Rook7th;
		int file = Col(f) + 1;
		if (pentry.m_ranks[file][WHITE] == 0)
		{
			if (pentry.m_ranks[file][BLACK] == 7)
				posScore += RookOpen;
			else
				posScore += RookSemiopen;
		}
	}

	x = pos.Bits(RB);
	while (x)
	{
		f = PopLSB(x);
		posScore -= ROOK_KING_TROPISM[Dist(f, pos.King(WHITE))];
		y = RookAttacks(f, occ ^ pos.Bits(RB));
		posScore -= ROOK_MOBILITY[CountBits(y)];
		posScore -= AttackStrongerPiece * CountBits(y & pos.Bits(QW));
		if (Row(f) == 6) posScore -= Rook7th;
		int file = Col(f) + 1;
		if (pentry.m_ranks[file][BLACK] == 7)
		{
			if (pentry.m_ranks[file][WHITE] == 0)
				posScore -= RookOpen;
			else
				posScore -= RookSemiopen;
		}
	}

	//
	//   QUEENS
	//

	x = pos.Bits(QW);
	while (x)
	{
		f = PopLSB(x);
		if (Row(f) == 1) posScore += Queen7th;
		posScore += QUEEN_KING_TROPISM[Dist(f, pos.King(BLACK))];
		posScore += PSQ_Q_END[f] * (32 - pos.MatIndex(BLACK)) / 32;
		y = QueenAttacks(f, occ ^ pos.Bits(RW));
	}

	x = pos.Bits(QB);
	while (x)
	{
		f = PopLSB(x);
		if (Row(f) == 6) posScore -= Queen7th;
		posScore -= QUEEN_KING_TROPISM[Dist(f, pos.King(WHITE))];
		posScore -= PSQ_Q_END[FLIP[f]] * (32 - pos.MatIndex(WHITE)) / 32;
		y = QueenAttacks(f, occ ^ pos.Bits(RB));
	}

	//
	//   KINGS
	//

	{
		f = pos.King(WHITE);
		posScore += PSQ_K_MID[f] * pos.MatIndex(BLACK) / 32;
		posScore += PSQ_K_END[f] * (32 - pos.MatIndex(BLACK)) / 32;
		int penalty = PawnShieldWhite(pentry, f);
		posScore += KING_PAWN_SHIELD[penalty] * pos.MatIndex(BLACK) / 32;
	}

	{
		f = pos.King(BLACK);
		posScore -= PSQ_K_MID[FLIP[f]] * pos.MatIndex(WHITE) / 32;
		posScore -= PSQ_K_END[FLIP[f]] * (32 - pos.MatIndex(WHITE)) / 32;
		int penalty = PawnShieldBlack(pentry, f);
		posScore -= KING_PAWN_SHIELD[penalty] * pos.MatIndex(WHITE) / 32;
	}

	U64 weak[2] =
	{
		LL(0x00000000ffffffff) & pentry.m_attacks[BLACK] & ~pentry.m_potentialAttacks[WHITE],
		LL(0xffffffff00000000) & pentry.m_attacks[WHITE] & ~pentry.m_potentialAttacks[BLACK]
	};

	//   weak fields
	posScore += WeakField * CountBits(weak[WHITE]);
	posScore -= WeakField * CountBits(weak[BLACK]);

	//   strong fields
	x = (pos.Bits(NW) | pos.Bits(BW)) & weak[BLACK];
	posScore += MinorOnStrongField * CountBits(x);

	x = (pos.Bits(NB) | pos.Bits(BB)) & weak[WHITE];
	posScore -= MinorOnStrongField * CountBits(x);

	EVAL e = matScore + posScore * g_positionalKnowledge / 100;

	if (e > 0 && ee == EVAL_WHITE_CANNOT_WIN)
		e = 0;

	if (e < 0 && ee == EVAL_BLACK_CANNOT_WIN)
		e = 0;

	if (ee == EVAL_PROBABLE_DRAW)
		e /= 2;

	return (pos.Side() == WHITE)? e : -e;
}

const int center[64] =
{
	-3, -2, -1,  0,  0, -1, -2, -3,
	-2, -1,  0,  1,  1,  0, -1, -2,
	-1,  0,  1,  2,  2,  1,  0, -1,
	 0,  1,  2,  3,  3,  2,  1,  0,
	 0,  1,  2,  3,  3,  2,  1,  0,
	-1,  0,  1,  2,  2,  1,  0, -1,
	-2, -1,  0,  1,  1,  0, -1, -2,
	-3, -2, -1,  0,  0, -1, -2, -3
};

const int pawn_center[64] =
{
	0,  0,  0,  0,  0,  0,  0,  0,
	0,  0,  0,  0,  0,  0,  0,  0,
	0,  0,  0,  1,  1,  0,  0,  0,
	0,  0,  0,  1,  1,  0,  0,  0,
	0,  0,  0,  1,  1,  0,  0,  0,
	0,  0,  0,  0,  0,  0,  0,  0,
	0,  0,  0,  0,  0,  0,  0,  0,
	0,  0,  0,  0,  0,  0,  0,  0
};

const int pawn_attack_center[64] =
{
	0,  0,  0,  0,  0,  0,  0,  0,
	0,  0,  0,  0,  0,  0,  0,  0,
	0,  0,  0,  0,  0,  0,  0,  0,
	0,  0,  1,  1,  1,  1,  0,  0,
	0,  0,  1,  1,  1,  1,  0,  0,
	0,  0,  1,  1,  1,  1,  0,  0,
	0,  0,  0,  0,  0,  0,  0,  0,
	0,  0,  0,  0,  0,  0,  0,  0
};

const int center_k[64] =
{
	4,  4,  4,  4,  4,  4,  4,  4,
	4,  4,  4,  4,  4,  4,  4,  4,
	4,  4,  4,  4,  4,  4,  4,  4,
	4,  4,  4,  4,  4,  4,  4,  4,
	4,  4,  4,  4,  4,  4,  4,  4,
	4,  4,  4,  4,  4,  4,  4,  4,
	2,  2,  2,  2,  2,  2,  2,  2,
	1,  0,  0,  1,  0,  1,  0,  1
};

void InitEval(const std::vector<int>& x)
{
	int i = 0;
	VAL_P = x[i++];
	VAL_N = x[i++];
	VAL_B = x[i++];
	VAL_R = x[i++];
	VAL_Q = x[i++];
	PawnEndgameBonus = x[i++];
	PawnDoubled = x[i++];
	PawnIsolated = x[i++];
	PawnCenter = x[i++];
	PawnPassed = x[i++];
	PawnPassedKingDist = x[i++];
	PawnPassedSquare = x[i++];
	PawnPassedBlocker = x[i++];
	KnightCenter = x[i++];
	KnightKingTropism = x[i++];
	BishopPairEndgame = x[i++];
	BishopMobility = x[i++];
	BishopKingTropism = x[i++];
	Rook7th = x[i++];
	RookSemiopen = x[i++];
	RookOpen = x[i++];
	RookMobility = x[i++];
	RookKingTropism = x[i++];
	QueenKingTropism = x[i++];
	QueenCenterEndgame = x[i++];
	Queen7th = x[i++];
	KingCenterMidgame = x[i++];
	KingCenterEndgame = x[i++];
	KingPawnShield = x[i++];
	AttackStrongerPiece = x[i++];
	WeakField = x[i++];
	MinorOnStrongField = x[i++];
	assert(i == NUM_WEIGHTS);

	for (int f = 0; f < 64; ++f)
	{
		PSQ_P[f] = PawnCenter * pawn_center[f];
		PSQ_N[f] = KnightCenter * center[f] / 3;
		PSQ_Q_END[f] = QueenCenterEndgame * center[f] / 3;
		PSQ_K_MID[f] = KingCenterMidgame * center_k[f] / 4;
		PSQ_K_END[f] = KingCenterEndgame * center[f] / 3;
	}

	for (int r = 0; r < 8; ++r)
		PAWN_PASSED[r] = (int)InterpolateSquare(r, 0, 7, 0, PawnPassed);
	for (int m = 0; m < 14; ++m)
		BISHOP_MOBILITY[m] = (int)InterpolateSquare(m, 13, 1, BishopMobility, -BishopMobility);
	for (int m = 0; m < 15; ++m)
		ROOK_MOBILITY[m] = (int)InterpolateSquare(m, 14, 2, RookMobility, -RookMobility);
	for (int d = 0; d < 10; ++d)
	{
		KNIGHT_KING_TROPISM[d] = (int)InterpolateLinear(d, 9, 1, 0, KnightKingTropism);
		BISHOP_KING_TROPISM[d] = (int)InterpolateLinear(d, 9, 1, 0, BishopKingTropism);
		ROOK_KING_TROPISM[d] = (int)InterpolateLinear(d, 9, 1, 0, RookKingTropism);
		QUEEN_KING_TROPISM[d] = (int)InterpolateLinear(d, 9, 1, 0, QueenKingTropism);
	}
	for (int p = 0; p < 10; ++p)
		KING_PAWN_SHIELD[p] = - (int)InterpolateSquare(p, 0, 9, 0, KingPawnShield);

	VALUE[0] = 0;
	VALUE[1] = 0;
	VALUE[2] = VAL_P;
	VALUE[3] = VAL_P;
	VALUE[4] = VAL_N;
	VALUE[5] = VAL_N;
	VALUE[6] = VAL_B;
	VALUE[7] = VAL_B;
	VALUE[8] = VAL_R;
	VALUE[9] = VAL_R;
	VALUE[10] = VAL_Q;
	VALUE[11] = VAL_Q;
	VALUE[12] = 0;
	VALUE[13] = 0;
}

void GetEvalWeights(std::vector<int>& x)
{
	x.clear();
	x.push_back(VAL_P);
	x.push_back(VAL_N);
	x.push_back(VAL_B);
	x.push_back(VAL_R);
	x.push_back(VAL_Q);
	x.push_back(PawnEndgameBonus);
	x.push_back(PawnDoubled);
	x.push_back(PawnIsolated);
	x.push_back(PawnCenter);
	x.push_back(PawnPassed);
	x.push_back(PawnPassedKingDist);
	x.push_back(PawnPassedSquare);
	x.push_back(PawnPassedBlocker);
	x.push_back(KnightCenter);
	x.push_back(KnightKingTropism);
	x.push_back(BishopPairEndgame);
	x.push_back(BishopMobility);
	x.push_back(BishopKingTropism);
	x.push_back(Rook7th);
	x.push_back(RookSemiopen);
	x.push_back(RookOpen);
	x.push_back(RookMobility);
	x.push_back(RookKingTropism);
	x.push_back(QueenKingTropism);
	x.push_back(QueenCenterEndgame);
	x.push_back(Queen7th);
	x.push_back(KingCenterMidgame);
	x.push_back(KingCenterEndgame);
	x.push_back(KingPawnShield);
	x.push_back(AttackStrongerPiece);
	x.push_back(WeakField);
	x.push_back(MinorOnStrongField);
	assert(x.size() == NUM_WEIGHTS);
}

void InitEval()
{
	std::vector<int> x;
	if (!LoadVector("weights.txt", x))
	{
		// create file with default values
		GetEvalWeights(x);
		if (SaveVector("weights.txt", x))
			out("Created file 'weights.txt' with default values\n");
		else
			out("Failed to create file 'weights.txt' with default values\n");
	}

	if (x.size() != NUM_WEIGHTS)
	{
		out("Incorrect number of parameters in weights.txt: %d instead of %d\n", x.size(), NUM_WEIGHTS);
		out("Using default values\n");
		GetEvalWeights(x);
	}
	else
	{
		// load values from file
		out("Loaded %d evaluation parameters from file 'weights.txt'\n", x.size());
	}

	InitEval(x);
}

void PawnEntry::Read(const Position& pos)
{
	U64 x, y;
	FLD f;
	int file, rank;

	m_pawnHash = pos.PawnHash();
	m_score = 0;
	m_passed[WHITE] = m_passed[BLACK] = 0;
	m_attacks[WHITE] = m_attacks[BLACK] = 0;
	m_potentialAttacks[WHITE] = m_potentialAttacks[BLACK] = 0;

	for (file = 0; file < 10; ++file)
	{
		m_ranks[file][WHITE] = 0;
		m_ranks[file][BLACK] = 7;
	}

	x = pos.Bits(PW);
	while (x)
	{
		f = PopLSB(x);
		file = Col(f) + 1;
		rank = Row(f);
		if (rank > m_ranks[file][WHITE])
			m_ranks[file][WHITE] = rank;

		m_attacks[WHITE] |= BB_PAWN_ATTACKS[f][WHITE];
		y = Left(BB_DIR[f][DIR_U]) | Right(BB_DIR[f][DIR_U]);
		m_potentialAttacks[WHITE] |= y;
	}

	x = pos.Bits(PB);
	while (x)
	{
		f = PopLSB(x);
		file = Col(f) + 1;
		rank = Row(f);
		if (rank < m_ranks[file][BLACK])
			m_ranks[file][BLACK] = rank;

		m_attacks[BLACK] |= BB_PAWN_ATTACKS[f][BLACK];
		y = Left(BB_DIR[f][DIR_D]) | Right(BB_DIR[f][DIR_D]);
		m_potentialAttacks[BLACK] |= y;
	}

	x = pos.Bits(PW);
	while (x)
	{
		f = PopLSB(x);
		file = Col(f) + 1;
		rank = Row(f);

		m_score += PSQ_P[f];
		if (m_ranks[file][BLACK] == 7)
		{
			if (m_ranks[file - 1][BLACK] >= rank && m_ranks[file + 1][BLACK] >= rank)
				m_passed[WHITE] |= BB_SINGLE[f];
		}

		if (rank != m_ranks[file][WHITE])
			m_score += PawnDoubled;

		if (m_ranks[file - 1][WHITE] == 0 && m_ranks[file + 1][WHITE] == 0)
			m_score += PawnIsolated;
	}

	x = pos.Bits(PB);
	while (x)
	{
		f = PopLSB(x);
		file = Col(f) + 1;
		rank = Row(f);

		m_score -= PSQ_P[FLIP[f]];
		if (m_ranks[file][WHITE] == 0)
		{
			if (m_ranks[file - 1][WHITE] <= rank && m_ranks[file + 1][WHITE] <= rank)
				m_passed[BLACK] |= BB_SINGLE[f];
		}

		if (rank != m_ranks[file][BLACK])
			m_score -= PawnDoubled;

		if (m_ranks[file - 1][BLACK] == 7 && m_ranks[file + 1][BLACK] == 7)
			m_score -= PawnIsolated;
	}
}

EVAL_ESTIMATION EstimateDraw(const Position& pos)
{
	static const int FIELD_COLOR[64] =
	{
		0, 1, 0, 1, 0, 1, 0, 1,
		1, 0, 1, 0, 1, 0, 1, 0,
		0, 1, 0, 1, 0, 1, 0, 1,
		1, 0, 1, 0, 1, 0, 1, 0,
		0, 1, 0, 1, 0, 1, 0, 1,
		1, 0, 1, 0, 1, 0, 1, 0,
		0, 1, 0, 1, 0, 1, 0, 1,
		1, 0, 1, 0, 1, 0, 1, 0
	};

	// KBP-K

	if (pos.MatIndex(WHITE) == 3 && pos.Count(PW) == 1)
	{
		if (pos.MatIndex(BLACK) == 0 && pos.Count(PB) == 0 && pos.Count(BW) == 1)
		{
			FLD pawn = LSB(pos.Bits(PW));
			FLD bishop = LSB(pos.Bits(BW));
			FLD king = pos.King(BLACK);
			if (Col(pawn) == 0)
			{
				if (FIELD_COLOR[A8] != FIELD_COLOR[bishop])
				{
					if (king == A8 || king == A7 || king == B8 || king == B7)
						return EVAL_PRACTICAL_DRAW;
				}
			}
			if (Col(pawn) == 7)
			{
				if (FIELD_COLOR[H8] != FIELD_COLOR[bishop])
				{
					if (king == H8 || king == H7 || king == G8 || king == G7)
						return EVAL_PRACTICAL_DRAW;
				}
			}
		}
	}

	if (pos.MatIndex(BLACK) == 3 && pos.Count(PB) == 1)
	{
		if (pos.MatIndex(WHITE) == 0 && pos.Count(PW) == 0 && pos.Count(BB) == 1)
		{
			FLD pawn = LSB(pos.Bits(PB));
			FLD bishop = LSB(pos.Bits(BB));
			FLD king = pos.King(WHITE);
			if (Col(pawn) == 0)
			{
				if (FIELD_COLOR[A1] != FIELD_COLOR[bishop])
				{
					if (king == A1 || king == A2 || king == B1 || king == B2)
						return EVAL_PRACTICAL_DRAW;
				}
			}
			if (Col(pawn) == 7)
			{
				if (FIELD_COLOR[H1] != FIELD_COLOR[bishop])
				{
					if (king == H1 || king == H2 || king == G1 || king == G2)
						return EVAL_PRACTICAL_DRAW;
				}
			}
		}
	}

	// KB-KB

	if (pos.MatIndex(WHITE) == 3 && pos.MatIndex(BLACK) == 3)
	{
		if (pos.Count(BW) == 1 && pos.Count(BB) == 1)
		{
			FLD bw = LSB(pos.Bits(BW));
			FLD bb = LSB(pos.Bits(BB));
			if (FIELD_COLOR[bw] != FIELD_COLOR[bb])
				return EVAL_PROBABLE_DRAW;
		}
	}

	// LOW MATERIAL

	// We can claim draw only in case of "King + minor vs. King".
	if (pos.Count(PW) == 0 && pos.Count(PB) == 0)
	{
		if (pos.MatIndex(WHITE) + pos.MatIndex(BLACK) <= 3)
			return EVAL_THEORETICAL_DRAW;
	}

	// Engames like "King + minor vs. King + minor" - just practical draw,
	// but no draw claim, because checkmate positions do exist.
	if (pos.Count(PW) == 0 && pos.MatIndex(WHITE) < 5)
	{
		if (pos.Count(PB) == 0 && pos.MatIndex(BLACK) < 5)
			return EVAL_PRACTICAL_DRAW;
		else
			return EVAL_WHITE_CANNOT_WIN;
	}

	if (pos.Count(PB) == 0 && pos.MatIndex(BLACK) < 5)
	{
		if (pos.Count(PW) == 0 && pos.MatIndex(WHITE) < 5)
			return EVAL_PRACTICAL_DRAW;
		else
			return EVAL_BLACK_CANNOT_WIN;
	}

	return EVAL_UNKNOWN;
}

bool SaveVector(const char* file, const std::vector<int>& x)
{
	FILE* dst = fopen(file, "wt");
	if (dst != NULL)
	{
		for (size_t i = 0; i < x.size(); ++i)
		{
			fprintf(dst, "%d", x[i]);
			if (i < x.size() - 1)
				fprintf(dst, " ");
		}
		fclose(dst);
		return true;
	}
	return false;
}

bool LoadVector(const char* file, std::vector<int>& x)
{
	x.clear();

	FILE* src = fopen(file, "rt");
	if (src != NULL)
	{
		char buf[4096];
		if (fgets(buf, sizeof(buf), src))
		{
			char* tk = strtok(buf, " ");
			while (tk)
			{
				size_t sz = strlen(tk);
				if (sz > 0 && (isdigit(tk[0]) || tk[0] == '-'))
					x.push_back(atoi(tk));
				tk = strtok(NULL, " ");
			}
		}
		fclose(src);
		return true;
	}
	return false;
}
