//  GREKO Chess Engine
//  (c) 2002-2016 Vladimir Medvedev <vrm@bk.ru>
//  http://greko.su

//  eval.cpp: static position evaluation
//  modified: 30-June-2016

#include "eval.h"
#include "utils.h"

EVAL VAL_P = 100;                  // 1
EVAL VAL_N = 400;                  // 2
EVAL VAL_B = 400;                  // 3
EVAL VAL_R = 600;                  // 4
EVAL VAL_Q = 1200;                 // 5

EVAL VALUE[14];

EVAL PawnDoubled          = -10;   // 6
EVAL PawnIsolated         = -19;   // 7
EVAL PawnBackwards        = -5;    // 8
EVAL PawnCenter           =  9;    // 9
EVAL PawnPassedFreeMax    = 128;   // 10
EVAL PawnPassedBlockedMax = 101;   // 11
EVAL PawnPassedKingDist   =   9;   // 12
EVAL PawnPassedSquare     = 200;   // 13
EVAL KnightCenter         =  27;   // 14
EVAL KnightOutpost        =   7;   // 15
EVAL KnightMobility       =  19;   // 16
EVAL BishopPairMidgame    =  20;   // 17
EVAL BishopPairEndgame    =  95;   // 18
EVAL BishopCenter         =   9;   // 19
EVAL BishopMobility       =  72;   // 20
EVAL Rook7th              =  24;   // 21
EVAL RookOpen             =  17;   // 22
EVAL RookMobility         =  40;   // 23
EVAL QueenKingTropism     =  99;   // 24
EVAL KingCenterMid        = -41;   // 25
EVAL KingCenterEnd        =  33;   // 26
EVAL KingPawnShield       = 120;   // 27

std::string paramNames[NUM_WEIGHTS] =
{
	"VAL_P",
	"VAL_N",
	"VAL_B",
	"VAL_R",
	"VAL_Q",
	"PawnDoubled",
	"PawnIsolated",
	"PawnBackwards",
	"PawnCenter",
	"PawnPassedFreeMax",
	"PawnPassedBlockedMax",
	"PawnPassedKingDist",
	"PawnPassedSquare",
	"KnightCenter",
	"KnightOutpost",
	"KnightMobility",
	"BishopPairMidgame",
	"BishopPairEndgame",
	"BishopCenter",
	"BishopMobility",
	"Rook7th",
	"RookOpen",
	"RookMobility",
	"QueenKingTropism",
	"KingCenterMid",
	"KingCenterEnd",
	"KingPawnShield"
};

int x_min[NUM_WEIGHTS] =
{
	0,      // VAL_P
	0,      // VAL_N
	0,      // VAL_B
	0,      // VAL_R
	0,      // VAL_Q
	0,      // PawnDoubled
	-100,   // PawnIsolated
	-100,   // PawnBackwards
	0,      // PawnCenter
	0,      // PawnPassedFreeMax
	0,      // PawnPassedBlockedMax
	0,      // PawnPassedKingDist
	0,      // PawnPassedSquare
	0,      // KnightCenter
	0,      // KnightOutpost
	0,      // KnightMobility
	0,      // BishopPairMidgame
	0,      // BishopPairEndgame
	0,      // BishopCenter
	0,      // BishopMobility
	0,      // Rook7th
	0,      // RookOpen
	0,      // RookMobility
	0,      // QueenKingTropism
	-100,   // KingCenterMid
	0,      // KingCenterEnd
	0       // KingPawnShield
};

int x_max[NUM_WEIGHTS] =
{
	2000,   // VAL_P
	2000,   // VAL_N
	2000,   // VAL_B
	2000,   // VAL_R
	2000,  // VAL_Q
	0,     // PawnDoubled
	0,     // PawnIsolated
	0,     // PawnBackwards
	200,   // PawnCenter
	200,   // PawnPassedFreeMax
	200,   // PawnPassedBlockedMax
	200,   // PawnPassedKingDist
	200,   // PawnPassedSquare
	200,   // KnightCenter
	200,   // KnightOutpost
	200,   // KnightMobility
	200,   // BishopPairMidgame
	200,   // BishopPairEndgame
	200,   // BishopCenter
	200,   // BishopMobility
	200,   // Rook7th
	200,   // RookOpen
	200,   // RookMobility
	200,   // QueenKingTropism
	0,     // KingCenterMid
	200,   // KingCenterEnd
	200    // KingPawnShield
};

std::string ParamName(int index)
{
	if (index < 0 || index >= NUM_WEIGHTS)
		return "(incorrect index)";
	else
		return paramNames[index];
}

EVAL PSQ_P[64];
EVAL PSQ_N[64];
EVAL PSQ_B[64];
EVAL PSQ_K_MID[64];
EVAL PSQ_K_END[64];

EVAL PAWN_PASSED_FREE[8];
EVAL PAWN_PASSED_BLOCKED[8];
EVAL KNIGHT_MOBILITY[9];
EVAL BISHOP_MOBILITY[14];
EVAL ROOK_MOBILITY[15];
EVAL QUEEN_KING_TROPISM[8];
EVAL KING_PAWN_SHIELD[10];

double Function2(double arg, double argFrom, double argTo, double valFrom, double valTo)
{
	// A*x*x + B*x + C

	double x = arg - argFrom;
	double xMax = argTo - argFrom;

	double A = (valTo - valFrom) / (xMax * xMax);
	double B = 0;
	double C = valFrom;

	return A * x * x + B * x + C;
}

struct PawnEntry
{
	U32  m_pawnHash;
	int  m_ranks[10][2];
	EVAL m_score;
	U64  m_passed[2];

	void Read(const Position& pos);
};

const int g_pawnHashSize = 8192;
PawnEntry g_pawnHash[g_pawnHashSize];
EvalParams g_evalParams;

inline int Dist(FLD f1, FLD f2)
{
	int drow = Row(f1) - Row(f2);
	if (drow < 0)
		drow = -drow;

	int dcol = Col(f1) - Col(f2);
	if (dcol < 0)
		dcol = -dcol;

	return (drow > dcol)? drow : dcol;
}

int PawnShieldWhite(const PawnEntry& pentry, FLD K)
{
	int r = 0;
	int file = Col(K) + 1;
	for (int i = file - 1; i <= file + 1; ++i)
	{
		int rank = pentry.m_ranks[i][WHITE];
		if (rank == 6)
			;
		else if (rank == 5)
			r += 1;
		else if (rank == 4)
			r += 2;
		else
			r += 3;
	}
	return r;
}

int PawnShieldBlack(const PawnEntry& pentry, FLD K)
{
	int r = 0;
	int file = Col(K) + 1;
	for (int i = file - 1; i <= file + 1; ++i)
	{
		int rank = pentry.m_ranks[i][BLACK];
		if (rank == 1)
			;
		else if (rank == 2)
			r += 1;
		else if (rank == 3)
			r += 2;
		else
			r += 3;
	}
	return r;
}

EVAL Evaluate(const Position& pos, EVAL alpha, EVAL beta)
{
	EVAL_ESTIMATION ee = EstimateDraw(pos);
	if (ee == EVAL_THEORETICAL_DRAW || ee == EVAL_PRACTICAL_DRAW)
		return DRAW_SCORE;

	EVAL matScore = pos.Material(WHITE) - pos.Material(BLACK);
	if (pos.Count(BW) == 2)
		matScore += (BishopPairMidgame * pos.MatIndex(BLACK) + BishopPairEndgame * (32 - pos.MatIndex(BLACK))) / 32;
	if (pos.Count(BB) == 2)
		matScore -= (BishopPairMidgame * pos.MatIndex(WHITE) + BishopPairEndgame * (32 - pos.MatIndex(WHITE))) / 32;
	matScore = matScore * g_evalParams.Material / 50;

	EVAL lazy = (pos.Side() == WHITE)? matScore : -matScore;
	if (lazy <= alpha - g_evalParams.LazyEvalMargin)
		return alpha;
	if (lazy >= beta + g_evalParams.LazyEvalMargin)
		return beta;

	EVAL posScore = 0;
	EVAL boardScore = 0;
	EVAL mobScore = 0;
	EVAL pawnStructScore = 0;
	EVAL pawnPassedScore = 0;
	EVAL kingSafetyScore = 0;
	U64 x, y, occ = pos.BitsAll();
	FLD f;

	//
	//   PAWNS
	//

	int index = pos.PawnHash() % g_pawnHashSize;

	PawnEntry& pentry = g_pawnHash[index];
	if (pentry.m_pawnHash != pos.PawnHash())
		pentry.Read(pos);

	pawnStructScore += pentry.m_score;

	x = pentry.m_passed[WHITE];
	while (x)
	{
		f = Bitboard::PopLSB(x);

		if (pos[f - 8] == NOPIECE)
			pawnPassedScore += PAWN_PASSED_FREE[7 - Row(f)];
		else
			pawnPassedScore += PAWN_PASSED_BLOCKED[7 - Row(f)];

		if (pos.MatIndex(BLACK) == 0)
		{
			FLD f1 = f;
			if (pos.Side() == BLACK)
				f1 += 8;
			if ((Bitboard::PawnSquare(f1, WHITE) & pos.Bits(KB)) == 0)
				pawnPassedScore += PawnPassedSquare * (7 - Row(f1)) / 6;
		}
		else if (pos.MatIndex(BLACK) < 10)
			pawnPassedScore += PawnPassedKingDist * Dist(f - 8, pos.King(BLACK));
	}

	x = pentry.m_passed[BLACK];
	while (x)
	{
		f = Bitboard::PopLSB(x);

		if (pos[f + 8] == NOPIECE)
			pawnPassedScore -= PAWN_PASSED_FREE[Row(f)];
		else
			pawnPassedScore -= PAWN_PASSED_BLOCKED[Row(f)];

		if (pos.MatIndex(WHITE) == 0)
		{
			FLD f1 = f;
			if (pos.Side() == WHITE)
				f1 -= 8;
			if ((Bitboard::PawnSquare(f1, BLACK) & pos.Bits(KW)) == 0)
				pawnPassedScore -= PawnPassedSquare * Row(f1) / 6;
		}
		else if (pos.MatIndex(WHITE) < 10)
			pawnPassedScore -= PawnPassedKingDist * Dist(f + 8, pos.King(WHITE));
	}

	//
	//   KNIGHTS
	//

	static const int outpost[64] =
	{
		0, 0, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 1, 1, 0, 0, 0,
		0, 0, 1, 1, 1, 1, 0, 0,
		0, 1, 1, 1, 1, 1, 1, 0,
		0, 0, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0
	};

	static const int knight_mob[64] =
	{
		2, 3, 4, 4, 4, 4, 3, 2,
		3, 4, 6, 6, 6, 6, 4, 3,
		4, 6, 8, 8, 8, 8, 6, 4,
		4, 6, 8, 8, 8, 8, 6, 4,
		4, 6, 8, 8, 8, 8, 6, 4,
		4, 6, 8, 8, 8, 8, 6, 4,
		3, 4, 6, 6, 6, 6, 4, 3,
		2, 3, 4, 4, 4, 4, 3, 2
	};

	x = pos.Bits(NW);
	while (x)
	{
		f = Bitboard::PopLSB(x);
		boardScore += PSQ_N[f];
		int mob = knight_mob[f];
		mobScore += KNIGHT_MOBILITY[mob];
		if (outpost[f])
		{
			if (Bitboard::PawnAttacks(f, BLACK) & pos.Bits(PW))
				boardScore += KnightOutpost;
		}
	}

	x = pos.Bits(NB);
	while (x)
	{
		f = Bitboard::PopLSB(x);
		boardScore -= PSQ_N[FLIP[f]];
		int mob = knight_mob[f];
		mobScore -= KNIGHT_MOBILITY[mob];
		if (outpost[FLIP[f]])
		{
			if (Bitboard::PawnAttacks(f, WHITE) & pos.Bits(PB))
				boardScore -= KnightOutpost;
		}
	}

	//
	//   BISHOPS
	//

	x = pos.Bits(BW);
	while (x)
	{
		f = Bitboard::PopLSB(x);
		boardScore += PSQ_B[f];
		y = Bitboard::BishopAttacks(f, occ);
		mobScore += BISHOP_MOBILITY[Bitboard::CountBits(y)];
	}

	x = pos.Bits(BB);
	while (x)
	{
		f = Bitboard::PopLSB(x);
		boardScore -= PSQ_B[FLIP[f]];
		y = Bitboard::BishopAttacks(f, occ);
		mobScore -= BISHOP_MOBILITY[Bitboard::CountBits(y)];
	}

	//
	//   ROOKS
	//

	x = pos.Bits(RW);
	while (x)
	{
		f = Bitboard::PopLSB(x);
		y = Bitboard::RookAttacks(f, occ ^ pos.Bits(RW));
		mobScore += ROOK_MOBILITY[Bitboard::CountBits(y)];

		if (Row(f) == 1)
			boardScore += Rook7th;

		int file = Col(f) + 1;
		if (pentry.m_ranks[file][WHITE] == 0)
			boardScore += RookOpen;
	}

	x = pos.Bits(RB);
	while (x)
	{
		f = Bitboard::PopLSB(x);
		y = Bitboard::RookAttacks(f, occ ^ pos.Bits(RB));
		mobScore -= ROOK_MOBILITY[Bitboard::CountBits(y)];

		if (Row(f) == 6)
			boardScore -= Rook7th;

		int file = Col(f) + 1;
		if (pentry.m_ranks[file][BLACK] == 7)
			boardScore -= RookOpen;
	}

	//
	//   QUEENS
	//

	x = pos.Bits(QW);
	while (x)
	{
		f = Bitboard::PopLSB(x);
		kingSafetyScore += QUEEN_KING_TROPISM[Dist(f, pos.King(BLACK))];
	}

	x = pos.Bits(QB);
	while (x)
	{
		f = Bitboard::PopLSB(x);
		kingSafetyScore -= QUEEN_KING_TROPISM[Dist(f, pos.King(WHITE))];
	}

	//
	//   KINGS
	//

	{
		f = pos.King(WHITE);
		kingSafetyScore += PSQ_K_MID[f] * pos.MatIndex(BLACK) / 32;
		boardScore += PSQ_K_END[f] * (32 - pos.MatIndex(BLACK)) / 32;
		int penalty = PawnShieldWhite(pentry, f);
		kingSafetyScore += KING_PAWN_SHIELD[penalty] * pos.MatIndex(BLACK) / 32;
	}

	{
		f = pos.King(BLACK);
		kingSafetyScore -= PSQ_K_MID[FLIP[f]] * pos.MatIndex(WHITE) / 32;
		boardScore -= PSQ_K_END[FLIP[f]] * (32 - pos.MatIndex(WHITE)) / 32;
		int penalty = PawnShieldBlack(pentry, f);
		kingSafetyScore -= KING_PAWN_SHIELD[penalty] * pos.MatIndex(WHITE) / 32;
	}

	posScore += mobScore * g_evalParams.Mobility / 50;
	posScore += pawnStructScore * g_evalParams.PawnStruct / 50;
	posScore += pawnPassedScore * g_evalParams.PawnPassed / 50;
	posScore += kingSafetyScore * g_evalParams.KingSafety / 50;
	posScore += boardScore * g_evalParams.BoardControl / 50;

	EVAL e = matScore + posScore;

	if (e > 0 && ee == EVAL_WHITE_CANNOT_WIN)
		e = 0;

	if (e < 0 && ee == EVAL_BLACK_CANNOT_WIN)
		e = 0;

	if (ee == EVAL_PROBABLE_DRAW)
		e /= 2;

	return (pos.Side() == WHITE)? e : -e;
}

const int center[64] =
{
	-3, -2, -1,  0,  0, -1, -2, -3,
	-2, -1,  0,  1,  1,  0, -1, -2,
	-1,  0,  1,  2,  2,  1,  0, -1,
	 0,  1,  2,  3,  3,  2,  1,  0,
	 0,  1,  2,  3,  3,  2,  1,  0,
	-1,  0,  1,  2,  2,  1,  0, -1,
	-2, -1,  0,  1,  1,  0, -1, -2,
	-3, -2, -1,  0,  0, -1, -2, -3
};

const int center_p[64] =
{
	0,  0,  0,  0,  0,  0,  0,  0,
	0,  0,  0,  0,  0,  0,  0,  0,
	0,  0,  0,  2,  2,  0,  0,  0,
	0,  0,  1,  2,  2,  1,  0,  0,
	0,  0,  1,  2,  2,  1,  0,  0,
	0,  0,  1,  1,  1,  1,  0,  0,
	0,  0,  0,  0,  0,  0,  0,  0,
	0,  0,  0,  0,  0,  0,  0,  0
};

const int center_k[64] =
{
	4,  4,  4,  4,  4,  4,  4,  4,
	4,  4,  4,  4,  4,  4,  4,  4,
	4,  4,  4,  4,  4,  4,  4,  4,
	4,  4,  4,  4,  4,  4,  4,  4,
	4,  4,  4,  4,  4,  4,  4,  4,
	4,  4,  4,  4,  4,  4,  4,  4,
	2,  2,  2,  2,  2,  2,  2,  2,
	1,  0,  0,  1,  0,  1,  0,  1
};

void InitEval(const std::vector<int>& x)
{
	int i = 0;
	VAL_P = x[i++];
	VAL_N = x[i++];
	VAL_B = x[i++];
	VAL_R = x[i++];
	VAL_Q = x[i++];
	PawnDoubled = x[i++];
	PawnIsolated = x[i++];
	PawnBackwards = x[i++];
	PawnCenter = x[i++];
	PawnPassedFreeMax = x[i++];
	PawnPassedBlockedMax = x[i++];
	PawnPassedKingDist = x[i++];
	PawnPassedSquare = x[i++];
	KnightCenter = x[i++];
	KnightOutpost = x[i++];
	KnightMobility = x[i++];
	BishopPairMidgame = x[i++];
	BishopPairEndgame = x[i++];
	BishopCenter = x[i++];
	BishopMobility = x[i++];
	Rook7th = x[i++];
	RookOpen = x[i++];
	RookMobility = x[i++];
	QueenKingTropism = x[i++];
	KingCenterMid = x[i++];
	KingCenterEnd = x[i++];
	KingPawnShield = x[i++];
	assert(i == NUM_WEIGHTS);

	for (int f = 0; f < 64; ++f)
	{
		PSQ_P[f] = PawnCenter * center_p[f] / 2;
		PSQ_N[f] = KnightCenter * center[f] / 3;
		PSQ_B[f] = BishopCenter * center[f] / 3;
		PSQ_K_MID[f] = KingCenterMid * center_k[f] / 4;
		PSQ_K_END[f] = KingCenterEnd * center[f] / 3;
	}

	for (int r = 0; r < 8; ++r)
	{
		PAWN_PASSED_FREE[r] = (int)Function2(r, 0, 7, 0, PawnPassedFreeMax);
		PAWN_PASSED_BLOCKED[r] = (int)Function2(r, 0, 7, 0, PawnPassedBlockedMax);
	}

	for (int m = 0; m < 9; ++m)
		KNIGHT_MOBILITY[m] = (int)Function2(m, 8, 2, KnightMobility, -KnightMobility);
	for (int m = 0; m < 14; ++m)
		BISHOP_MOBILITY[m] = (int)Function2(m, 13, 1, BishopMobility, -BishopMobility);
	for (int m = 0; m < 15; ++m)
		ROOK_MOBILITY[m] = (int)Function2(m, 14, 2, RookMobility, -RookMobility);
	for (int d = 0; d < 8; ++d)
		QUEEN_KING_TROPISM[d] = (int)Function2(d, 7, 2, 0, QueenKingTropism);
	for (int p = 0; p < 10; ++p)
		KING_PAWN_SHIELD[p] = - (int)Function2(p, 0, 9, 0, KingPawnShield);

	VALUE[0] = 0;
	VALUE[1] = 0;
	VALUE[2] = VAL_P;
	VALUE[3] = VAL_P;
	VALUE[4] = VAL_N;
	VALUE[5] = VAL_N;
	VALUE[6] = VAL_B;
	VALUE[7] = VAL_B;
	VALUE[8] = VAL_R;
	VALUE[9] = VAL_R;
	VALUE[10] = VAL_Q;
	VALUE[11] = VAL_Q;
	VALUE[12] = 0;
	VALUE[13] = 0;
}

void GetEvalWeights(std::vector<int>& x)
{
	x.clear();
	x.push_back(VAL_P);
	x.push_back(VAL_N);
	x.push_back(VAL_B);
	x.push_back(VAL_R);
	x.push_back(VAL_Q);
	x.push_back(PawnDoubled);
	x.push_back(PawnIsolated);
	x.push_back(PawnBackwards);
	x.push_back(PawnCenter);
	x.push_back(PawnPassedFreeMax);
	x.push_back(PawnPassedBlockedMax);
	x.push_back(PawnPassedKingDist);
	x.push_back(PawnPassedSquare);
	x.push_back(KnightCenter);
	x.push_back(KnightOutpost);
	x.push_back(KnightMobility);
	x.push_back(BishopPairMidgame);
	x.push_back(BishopPairEndgame);
	x.push_back(BishopCenter);
	x.push_back(BishopMobility);
	x.push_back(Rook7th);
	x.push_back(RookOpen);
	x.push_back(RookMobility);
	x.push_back(QueenKingTropism);
	x.push_back(KingCenterMid);
	x.push_back(KingCenterEnd);
	x.push_back(KingPawnShield);
	assert(x.size() == NUM_WEIGHTS);
}

void InitEval()
{
	std::vector<int> x;
	if (!LoadVector("weights.txt", x))
	{
		// create file with default values
		GetEvalWeights(x);
		if (SaveVector("weights.txt", x))
			out("Created file 'weights.txt' with default values\n");
		else
			out("Failed to create file 'weights.txt' with default values\n");
	}

	if (x.size() != NUM_WEIGHTS)
	{
		out("Incorrect number of parameters in weights.txt: %d instead of %d\n", x.size(), NUM_WEIGHTS);
		out("Using default values\n");
		GetEvalWeights(x);
	}
	else
	{
		// load values from file
		out("Loaded %d evaluation parameters from file 'weights.txt'\n", x.size());
	}

	InitEval(x);
}

void PawnEntry::Read(const Position& pos)
{
	U64 x;
	FLD f;
	int file, rank;

	m_pawnHash = pos.PawnHash();
	m_score = 0;
	m_passed[WHITE] = m_passed[BLACK] = 0;

	for (file = 0; file < 10; ++file)
	{
		m_ranks[file][WHITE] = 0;
		m_ranks[file][BLACK] = 7;
	}

	x = pos.Bits(PW);
	while (x)
	{
		f = Bitboard::PopLSB(x);
		file = Col(f) + 1;
		rank = Row(f);
		if (rank > m_ranks[file][WHITE])
			m_ranks[file][WHITE] = rank;
	}

	x = pos.Bits(PB);
	while (x)
	{
		f = Bitboard::PopLSB(x);
		file = Col(f) + 1;
		rank = Row(f);
		if (rank < m_ranks[file][BLACK])
			m_ranks[file][BLACK] = rank;
	}

	x = pos.Bits(PW);
	while (x)
	{
		f = Bitboard::PopLSB(x);
		file = Col(f) + 1;
		rank = Row(f);

		m_score += PSQ_P[f];
		if (m_ranks[file][BLACK] == 7)
		{
			if (m_ranks[file - 1][BLACK] >= rank && m_ranks[file + 1][BLACK] >= rank)
				m_passed[WHITE] |= Bitboard::Single(f);
		}

		if (rank != m_ranks[file][WHITE])
			m_score += PawnDoubled;

		if (m_ranks[file - 1][WHITE] == 0 && m_ranks[file + 1][WHITE] == 0)
			m_score += PawnIsolated;
		else if (m_ranks[file - 1][WHITE] < rank && m_ranks[file + 1][WHITE] < rank)
			m_score += PawnBackwards;
	}

	x = pos.Bits(PB);
	while (x)
	{
		f = Bitboard::PopLSB(x);
		file = Col(f) + 1;
		rank = Row(f);

		m_score -= PSQ_P[FLIP[f]];
		if (m_ranks[file][WHITE] == 0)
		{
			if (m_ranks[file - 1][WHITE] <= rank && m_ranks[file + 1][WHITE] <= rank)
				m_passed[BLACK] |= Bitboard::Single(f);
		}

		if (rank != m_ranks[file][BLACK])
			m_score -= PawnDoubled;

		if (m_ranks[file - 1][BLACK] == 7 && m_ranks[file + 1][BLACK] == 7)
			m_score -= PawnIsolated;
		else if (m_ranks[file - 1][BLACK] > rank && m_ranks[file + 1][BLACK] > rank)
			m_score -= PawnBackwards;
	}
}

EVAL_ESTIMATION EstimateDraw(const Position& pos)
{
	static const int FIELD_COLOR[64] =
	{
		0, 1, 0, 1, 0, 1, 0, 1,
		1, 0, 1, 0, 1, 0, 1, 0,
		0, 1, 0, 1, 0, 1, 0, 1,
		1, 0, 1, 0, 1, 0, 1, 0,
		0, 1, 0, 1, 0, 1, 0, 1,
		1, 0, 1, 0, 1, 0, 1, 0,
		0, 1, 0, 1, 0, 1, 0, 1,
		1, 0, 1, 0, 1, 0, 1, 0
	};

	// KBP-K

	if (pos.MatIndex(WHITE) == 3 && pos.Count(PW) == 1)
	{
		if (pos.MatIndex(BLACK) == 0 && pos.Count(PB) == 0 && pos.Count(BW) == 1)
		{
			FLD pawn = Bitboard::LSB(pos.Bits(PW));
			FLD bishop = Bitboard::LSB(pos.Bits(BW));
			FLD king = pos.King(BLACK);
			if (Col(pawn) == 0)
			{
				if (FIELD_COLOR[A8] != FIELD_COLOR[bishop])
				{
					if (king == A8 || king == A7 || king == B8 || king == B7)
						return EVAL_PRACTICAL_DRAW;
				}
			}
			if (Col(pawn) == 7)
			{
				if (FIELD_COLOR[H8] != FIELD_COLOR[bishop])
				{
					if (king == H8 || king == H7 || king == G8 || king == G7)
						return EVAL_PRACTICAL_DRAW;
				}
			}
		}
	}

	if (pos.MatIndex(BLACK) == 3 && pos.Count(PB) == 1)
	{
		if (pos.MatIndex(WHITE) == 0 && pos.Count(PW) == 0 && pos.Count(BB) == 1)
		{
			FLD pawn = Bitboard::LSB(pos.Bits(PB));
			FLD bishop = Bitboard::LSB(pos.Bits(BB));
			FLD king = pos.King(WHITE);
			if (Col(pawn) == 0)
			{
				if (FIELD_COLOR[A1] != FIELD_COLOR[bishop])
				{
					if (king == A1 || king == A2 || king == B1 || king == B2)
						return EVAL_PRACTICAL_DRAW;
				}
			}
			if (Col(pawn) == 7)
			{
				if (FIELD_COLOR[H1] != FIELD_COLOR[bishop])
				{
					if (king == H1 || king == H2 || king == G1 || king == G2)
						return EVAL_PRACTICAL_DRAW;
				}
			}
		}
	}

	// KB-KB

	if (pos.MatIndex(WHITE) == 3 && pos.MatIndex(BLACK) == 3)
	{
		if (pos.Count(BW) == 1 && pos.Count(BB) == 1)
		{
			FLD bw = Bitboard::LSB(pos.Bits(BW));
			FLD bb = Bitboard::LSB(pos.Bits(BB));
			if (FIELD_COLOR[bw] != FIELD_COLOR[bb])
				return EVAL_PROBABLE_DRAW;
		}
	}

	// LOW MATERIAL

	// We can claim draw only in case of "King + minor vs. King".
	if (pos.Count(PW) == 0 && pos.Count(PB) == 0)
	{
		if (pos.MatIndex(WHITE) + pos.MatIndex(BLACK) <= 3)
			return EVAL_THEORETICAL_DRAW;
	}

	// Engames like "King + minor vs. King + minor" - just practical draw,
	// but no draw claim, because checkmate positions do exist.
	if (pos.Count(PW) == 0 && pos.MatIndex(WHITE) < 5)
	{
		if (pos.Count(PB) == 0 && pos.MatIndex(BLACK) < 5)
			return EVAL_PRACTICAL_DRAW;
		else
			return EVAL_WHITE_CANNOT_WIN;
	}

	if (pos.Count(PB) == 0 && pos.MatIndex(BLACK) < 5)
	{
		if (pos.Count(PW) == 0 && pos.MatIndex(WHITE) < 5)
			return EVAL_PRACTICAL_DRAW;
		else
			return EVAL_BLACK_CANNOT_WIN;
	}

	return EVAL_UNKNOWN;
}

bool SaveVector(const char* file, const std::vector<int>& x)
{
	FILE* dst = fopen(file, "wt");
	if (dst != NULL)
	{
		for (size_t i = 0; i < x.size(); ++i)
		{
			fprintf(dst, "%d", x[i]);
			if (i < x.size() - 1)
				fprintf(dst, " ");
		}
		fclose(dst);
		return true;
	}
	return false;
}

bool LoadVector(const char* file, std::vector<int>& x)
{
	x.clear();

	FILE* src = fopen(file, "rt");
	if (src != NULL)
	{
		char buf[4096];
		if (fgets(buf, sizeof(buf), src))
		{
			char* tk = strtok(buf, " ");
			while (tk)
			{
				size_t sz = strlen(tk);
				if (sz > 0 && (isdigit(tk[0]) || tk[0] == '-'))
					x.push_back(atoi(tk));
				tk = strtok(NULL, " ");
			}
		}
		fclose(src);
		return true;
	}
	return false;
}

