		/**\
		 	Program file "gen.c"		
		 					
		 	      Public			
		 	       functions: generator()	
		 			  generator_i()	
		 			  generate_all()
		\**/

#include <mem.h>			/* memcpy() , NULL */
#include <stdlib.h>			/* qsort()  , NULL */

#include "lang.h"
#include "chess.h"
#include "global.h"
#include "current.h"
#include "proc.h"
#include "ks_macro.h"
#include "gener.h"
#include "attacks.h"
#include "experim.h"
#include "hashpos.h"
#include "ks_game.h"

static PIECE_IN_LIST *enemy_piece_list;
				/* Part of piece list with enemy pieces */
static PIECE_IN_LIST *our_piece_list;
				/* Part of piece list with our pieces */
PIECE_COLOR Enemy_color;	/* Color of our pieces */
PIECE_COLOR Our_color;		/* Color of enemy pieces */
MAT_EVAL mat_deficit;		/* Material deficit */
static MAT_EVAL near material_deficit(void);/* Compute material deficit */
static MAT_EVAL gen_check_flag=0;    /* Are we in gen_check ? */
static MAT_EVAL defence_flag=0;    /* Are we in defence ? */
static MAT_EVAL curr_profit;
static MAT_EVAL next_profit;



			/* Prototypes of generator functions */
static GEN_FUNCTION gen_check_answer;		/* Generate answers for check */
static GEN_FUNCTION gen_check_answer_and_killer;		/* Generate answers for check */
static GEN_FUNCTION gen_profitable_capture;	/* Generate profitable captures */
static GEN_FUNCTION gen_check_profitable_answer; /* Generate profitable answers to check */
static GEN_FUNCTION gen_capture;		/* Generate captures */
static GEN_FUNCTION gen_capture_and_killer;		/* Generate captures */
extern GEN_FUNCTION gen_normal_killer;		 	/* Generate killer moves */
extern void forced_killer(MOVE *first, MOVE *last);	/* Generate killer moves */
						  /* is in file "killer.c" */
static GEN_FUNCTION gen_castles;		/* Generate castles */
static GEN_FUNCTION gen_promotion;		/* Generate promotions */
static GEN_FUNCTION gen_check;			/* Generate checks */
static GEN_FUNCTION gen_capture_and_check_and_killer;	/* Generate capture and checks */
static GEN_FUNCTION gen_castle_check_and_all;	/* Generate castle check and all */
static GEN_FUNCTION gen_all;			/* Generate all moves */
static GEN_FUNCTION gen_passage;
static GEN_FUNCTION gen_profitable_attack;      
static GEN_FUNCTION gen_attack_answer;
static GEN_FUNCTION gen_nullmove;
static GEN_FUNCTION gen_attack_escape;   /* If win material escape from attack */
static NEAR void gen_pawn_one_move(SQUARE_INFO *fromp,SQUARE_INFO *top,
		WORD pm,MOVEFLAGS flag);

MOVE *movep;				/* Current move pointer */

typedef struct {			/* Define type element of GEN_STACK */
  GEN_FUNCTION_PTR *gen_func;		/* Pointer to generator function */
					  /* pointer to next generation */
}	GEN_STACK;
static GEN_STACK gen_stack[MAXDEPTH];	/* Generator stack */

typedef struct {		/* Define type PM */
  WORD mask;				/* piece mask */
  void *next;				/* pointer to the next such structure */
} PM;

static PM pm_queen_bishop,pm_queen_rook;
static PM pm_queen_bishop={0,&pm_queen_rook},	/* Our queen_bishop mask */
	  pm_queen_rook={0,&pm_queen_bishop};	/* Our queen_rook mask */

int move_cmp(const void *src1, const void *src2);
				/* Compare moves for sorting */
				/* This function cannot have modifier "near" */

void NEAR move_sort(MOVE *first,MOVE *last )
{
/*  killer(); */
  if( (first+1)==(last-1) ) {
	if( first->profit < (last-1)->profit )
	  swap_moves( first, last-1 );
  }
  else
	qsort(first,last-first,sizeof(MOVE),move_cmp);/* Sort written moves */
}

static MOVE * near legal_control(MOVE *first, MOVE *last)
				/* Check moves for validity */
				/* Return pointer to last+1 move */
{
  MOVE *cur;			/* Store current move pointer in position */
  MOVE *mp;			/* Current move pointer */

    cur=CURRENT;			/* Store current move pointer in position */
    for(mp=first; mp!=last; mp++) {	/* Move loop */
      CURRENT=mp;			/* Set move pointer for move() */
      if(move())			/* If move is not legal */
	CURRENT->profit=LARGENEG; /* assign profit maximum negative value */
      move_back();			/* Move back */
    }
    CURRENT=cur;			/* Restore current move pointer in position */
    if(last>first+1) move_sort(first, last );
    for(mp=first; mp!=last; mp++)	/* Move loop */
      if(mp->profit==LARGENEG)
	break;				/* Next moves are not legal */
    return(mp);
}

/*-------------------------------------------------------------------*/
		/* Arrays of generator functions pointers for different cases */
static GEN_FUNCTION_PTR gen_functions_all[] = {
				/* for function generate_all() */
  gen_profitable_capture,		/* Generate profitable captures */
					  /* and capture en passant */
  gen_passage,
  gen_castles,				/* Generate castles */
#ifdef DEFENCES
  gen_attack_answer,
#endif
  gen_all,				/* Generate all moves */
  NULL
};
static GEN_FUNCTION_PTR gen_functions_check[] = {
				/* CHECK state */
  gen_check_profitable_answer, /* Generate profitable answers to check */
  gen_check_answer_and_killer,			/* Generate answers for check */
  NULL
};
static GEN_FUNCTION_PTR gen_functions_normal[] = {
				/* status NORMAL */
  gen_attack_escape,   /* If win material escape from attack */
  gen_profitable_capture,  	/* Generate profitable captures */
  gen_passage,
#ifdef DEFENCES
  gen_attack_answer,
#endif
  gen_normal_killer,			 	/* Generate killer moves */
  gen_castle_check_and_all,		/* Generate castle check and all */
  NULL
};
static GEN_FUNCTION_PTR gen_functions_forced[] =  {
				/* status FORCED */
  gen_profitable_capture,		/* Generate profitable captures */
  gen_passage,
  gen_capture_and_check_and_killer,    	/* Generate captures and checks */
  NULL
};
static GEN_FUNCTION_PTR gen_functions_forced_without_check[] = {
				/* status FORCED WITHOUT_CHECK */
  gen_profitable_capture,		/* Generate profitable captures */
  gen_passage,
  gen_capture_and_killer,		/* Generate captures */
  NULL
};

extern int flPassGen;
int NEAR PassagePawn(void);
void NEAR calc_pawns_lines(void);

MOVE * generate_all(MOVE *mp)
			/* Generate all legal moves from first pass */
			/* for interface and fv0() and fv1() use only */
{
  MOVE *first;			/* First written move pointer */
  GEN_STACK *g_sp;		/* Generator stack pointer */

	 first=mp;				/* Store first written move pointer */
	 g_sp=&gen_stack[level];		/* Compute current generator stack pointer */
					/* Determine first pass generator function */
	 if(CHECK)
		g_sp->gen_func=gen_functions_check;
	 else /* G_STATUS==NORMAL */
		g_sp->gen_func=gen_functions_all;


	clear_moves();			/* Clear written moves masks */
	if(flPassGen){ calc_pawns_lines(); PassagePawn(); }

    movep=mp;				/* First written move pointer */
    do {				/* Do while generator generate something */
      mp=movep;				/* Store move pointer before generator call */
      generator(mp,1);			/* Call next pass of generator */
    } while(movep!=mp);
    return(legal_control(first,movep));	/* Control validity of moves */
}

MOVE * NEAR generator_i(MOVE *mp, int pass_en)
			/* Set first pass gen function and call generator */
{
  GEN_STACK *g_sp;		/* Generator stack pointer */

    g_sp=&gen_stack[level];	/* Compute current generator stack pointer */
				/* Determine first pass generator function */

    mat_deficit=material_deficit();	/* Compute material deficit */

    if(CHECK)				/* Answer for check */
      g_sp->gen_func=gen_functions_check;
    else if(G_STATUS&G_WITHOUT_CHECK){
	g_sp->gen_func=gen_functions_forced_without_check;
    }
    else if(G_STATUS&G_FORCED){
	g_sp->gen_func=gen_functions_forced;
    }
    else /* G_STATUS==NORMAL */
      g_sp->gen_func=gen_functions_normal;

    if( gen_hash_success ) gen_hash_success=0;/*move from hash */
    else mp=generator(mp, pass_en);
    return(mp);		/* Call generator */
}

MOVE * NEAR generator_null(MOVE *mp)
			/* Set first pass gen function and call generator */
{
/*  GEN_STACK *g_sp;		/* Generator stack pointer */

    g_sp=&gen_stack[level];	/* Compute current generator stack pointer */
				/* Determine first pass generator function */
    g_sp->gen_func=gen_functions_null;
    mp=generator(mp);
*/    return(mp);
}

MOVE * NEAR generator(MOVE *mp, int pass_en)
				/* Generate moves for current pass */
				/* Generator also defines next pass needed */
				/* If it return movep==mp then */
				  /* no more avaliable passes */
{
  GEN_FUNCTION_PTR **gen_function; /* Pointer to generator functions */
				     /* array element pointer */
    movep=mp;				/* Move pointer to generation */
					/* Take pointers to parts of piece list : */
    our_piece_list=&PIECE_LIST[color_to_king_num(Our_color)]; /* our pieces */
    enemy_piece_list=&PIECE_LIST[color_to_king_num(Enemy_color)]; /* enemy */

    mat_deficit=material_deficit();	/* Compute material deficit */

    gen_function=&gen_stack[level].gen_func;	/* Take pointer to current */
					  /* generator function pointer */
    while((**gen_function)!=NULL) {	/* Do generator function exist ? */
      (**gen_function)();		/* Call current generator function */
      (*gen_function)++;		/* Take next generator function */
      if(pass_en){
	if(mp!=movep) {			/* If generate something then return */
	  break;
	}
      }
    }
    if(mp+1<movep ) move_sort(mp,movep);/* Sort written moves */
    return(movep);
}

/*-------------------------------------------------------------------*/
			/* Internal generator functions prototypes */
static void near en_passant(void);
				/* Generate capture en passant */
static void near king_escape(SQUARE_NUM from, SQUARE_INFO *fromp, BYTE dir_mask);
				/* King escape from check in */
				  /* given directions without captures. */
				/* Generate capture en passant */
static void near go_to_square(SQUARE_NUM to, SQUARE_INFO *top, WORD piece_mask, WORD profitable);
				/* Write moves on given square by */
				  /* given set of pieces. */
				/* EXEPTION: it don't write pawns' vertical */
				  /* moves ! */
static void near pawn_go_to(SQUARE_NUM to, SQUARE_INFO *top);
				/* Write pawn vertical moves on square by */
				  /* all our pawns. */
void NEAR go_from_square(SQUARE_NUM from, SQUARE_INFO *fromp, BYTE dir_mask);
				/* Write moves by piece from square in */
				  /* given directions. */
static void near queen_go_from(SQUARE_NUM from, SQUARE_INFO *fromp, BYTE dir_mask);
				/* Write moves by queen-like piece from */
				  /* square in given directions. */
static void near knight_go_from(SQUARE_NUM from, SQUARE_INFO *fromp);
				/* Write moves by knight from square in */
				  /* all directions. */

static void near pawn_go_from(SQUARE_NUM from, SQUARE_INFO *fromp, BYTE dir_mask, MOVEFLAGS flag);
				/* Write moves by pawn from square in */
				  /* given directions. */

static void near attack_square(SQUARE_NUM en_sq, WORD avmask);
static void near defence_from_attack(SQUARE_NUM our_sq);
static MAT_EVAL move_profit(WORD our_att,WORD enemy_att,
      MAT_EVAL our_cost, MAT_EVAL capt_piece_cost,
      SQUARE_INFO *fromp,SQUARE_INFO *top);

/*-------------------------------------------------------------------*/
			/* Generator functions */

static GEN_FUNCTION gen_capture_and_check_and_killer
{
MOVE *mp;
  mp=movep;
  gen_check();
  gen_capture();
  forced_killer(mp,movep);
}
static GEN_FUNCTION gen_castle_check_and_all
{
    gen_castles();
    gen_check();
/*    gen_attack_answer(); */
    gen_all();
}
static GEN_FUNCTION gen_check_answer_and_killer
{
MOVE *mp;
  mp=movep;
  gen_check_answer();
  forced_killer(mp,movep);
}
static GEN_FUNCTION gen_capture_and_killer
{
MOVE *mp;
  mp=movep;
  gen_capture();
  forced_killer(mp,movep);
}

static GEN_FUNCTION gen_nullmove
{
  form_nullmove(movep++);
}

extern int flPassGen;
int NEAR PassagePawn(void);
extern SQUARE_NUM passage[NCOLOR][NLINE+1];

static GEN_FUNCTION gen_passage
{
SQUARE_NUM *sqp;
  if(!flPassGen) return;
  for(sqp=pos_sp->n.passage; *sqp!=DUMMY; sqp++)
	pawn_go_from(*sqp, SQUAREP(*sqp), 0xFF, M_PASSAGE);
}

static GEN_FUNCTION gen_check_answer	/* Generate answers for check */
{
  SQUARE_NUM king;		/* Our king square */
  SQUARE_INFO *kingp;	/* Our king square information */
  SQUARE_NUM to;		/* Square coordinate for move */
  SQUARE_INFO *top;		/* Pointer to current square information */
  PIECE_IN_LIST *enemy_piecep;/* Pointer to current enemy piece */
  WORD pm;			/* Current our piece mask */
  WORD check;			/* Piece mask for our king square */
  int step;			/* Coordinate step */
  BYTE check_dir;		/* Checking directions */
  int check_counter;		/* Check state indicator */

    en_passant();			/* Generate capture en passant */
    king=our_piece_list[0].where;	/* Our king coordinate */
    kingp=SQUAREP(king);		/* Our king square information pointer */
					/* Initialize enemy piece loop */
    check=kingp->attackedby.word[Enemy_color];	/* Who attack our king */
    enemy_piecep=&enemy_piece_list[1];	/* Skip king */
    pm=0x0002;				/* Enemy piece mask */
    check_dir=NIL;			/* Check directions mask */
    check_counter=bit_quantity[*((BYTE*)&check)]+
		  bit_quantity[*((BYTE*)&check+1)];
    if(check_counter<2) {		/* Not double check */
      do {				/* Enemy piece loop */
	if(check&pm) {  		/* Is current enemy piece attacking our king ? */
	  to=enemy_piecep->where;	/* Checking piece coordinate */
	  top=SQUAREP(to);		/* Checking piece square information */
	  switch(top->wood.name) {	/* Store check direction */
	    case PAWN:
	    case KNIGHT: step=0;	/* We can't defence king */
			 break; 	/* No unavaliable direction for king */
	    default:     check_dir|=direction[to][king];	/* Check direction mask */
			 step=increment[reverse_dir((byte_to_int(check_dir)))];
					/* Take coordinate increment for king defence */
	  }
	  break;			/* No more check pieces */
	}
      } while(				/* End of enemy piece loop */
	  enemy_piecep++, 		/* Take next enemy piece pointer */
	  pm<<=1 	 			/* Take next piece mask */
	);				/* Do while mask not empty */
      if(top->wood.cost>=mat_deficit)	/* Can we capture ? */
	go_to_square(to,top,~0x0000,0);
		/* Capture checking piece by all our pieces */

      if(!G_STATUS||mat_deficit<=0) {	/* Is status NORMAL ? */
	if(step) {			/* Can we defence of our king ? */
	  while((to-=step)!=king) {	/* Loop for squares between */
					  /* checking piece and king */
	    top-=step;			/* Take next square information pointer */
	    go_to_square(to,top,~0x0001,0);/* Go to this squares by all */
					  /* our pieces excluding king */
	    pawn_go_to(to,top);		/* Add vertical pawn moves */
	  }
	}
      }
    }
					/* King escapes from check in any */
					  /* direction excluding checking direction */
    king_escape(king,kingp,~check_dir);
}


static GEN_FUNCTION gen_capture		/* Generate captures */
{
  SQUARE_NUM to;		/* Square coordinate for move */
  SQUARE_INFO *top;		/* Pointer to square information */
  PIECE_IN_LIST *enemy_piecep;	/* Pointer to current enemy piece */
  WORD pm;			/* Current our piece mask */

    en_passant();			/* Generate capture en passant */
					/* Initialize enemy piece loop */
    enemy_piecep=&enemy_piece_list[1];	/* Skip king */
    pm=0x0002;				/* Enemy piece mask */
    do {				/* Enemy pieces loop */
      to=enemy_piecep->where;		/* Take enemy piece coordinate */
      if(to!=DUMMY) {			/* Is enemy piece exist in piece list ? */
	top=SQUAREP(to);		/* Square information pointer */
	if(top->wood.cost>=mat_deficit)	/* Can we capture this piece ? */
	  go_to_square(to,top,0xFFFF,0);  /* Write attacks of all our pieces to */
					  /* the square */
      }
    } while(				/* End of enemy piece loop */
	enemy_piecep++, 		/* Take next enemy piece pointer */
	pm<<=1 	 			/* Take next piece mask */
      );				/* Do while mask not empty */
}

static GEN_FUNCTION gen_profitable_capture	/* Generate profitable captures */
{
  SQUARE_NUM to;		/* Square coordinate for move */
  SQUARE_INFO *top;		/* Pointer to square information */
  PIECE_IN_LIST *enemy_piecep;	/* Pointer to current enemy piece */
  WORD pm;			/* Current our piece mask */

    gen_promotion();
    en_passant();			/* Generate capture en passant */
					/* Initialize enemy piece loop */
    enemy_piecep=&enemy_piece_list[1];	/* Skip king */
    pm=0x0002;				/* Enemy piece mask */
    do {				/* Enemy pieces loop */
      to=enemy_piecep->where;		/* Take enemy piece coordinate */
      if(to!=DUMMY) {			/* Is enemy piece exist in piece list ? */
	top=SQUAREP(to);		/* Square information pointer */
	if(top->wood.cost>=mat_deficit)	/* Can we capture this piece ? */
	  go_to_square(to,top,0xFFFF,1);  /* Write attacks of all our pieces to */
					  /* the square */
      }
    } while(				/* End of enemy piece loop */
	enemy_piecep++, 		/* Take next enemy piece pointer */
	pm<<=1 	 			/* Take next piece mask */
      );				/* Do while mask not empty */
}

static GEN_FUNCTION gen_check_profitable_answer /* Generate profitable answers to check*/
{
  SQUARE_NUM to;		/* Square coordinate for move */
  SQUARE_INFO *top;		/* Pointer to square information */

    en_passant();			/* Generate capture en passant */
					/* Initialize enemy piece loop */
	to=PREVIOUS->to;		/* Take enemy piece coordinate */
	top=SQUAREP(to);		/* Square information pointer */
	if(top->wood.cost>=mat_deficit)	/* Can we capture this piece ? */
	  go_to_square(to,top,0xFFFF,1);  /* Write attacks of all our pieces to */
					  /* the square */
}

static GEN_FUNCTION gen_castles		/* Generate castles */
{
  SQUARE_NUM line;		/* Begin of first/last line coordinate */
  SQUARE_INFO *linep;		/* Pointer to square information */

    if(CHECK) return;			/* No castles on check state */
    line=(Our_color==WHITE_PIECE)? a1: a8;/* Take begin of line */
    linep=SQUAREP(line);		/* Take pointer to square informtion */
    if(CASTLE[Our_color] & CASTLERIGHT) {  /* Consider right castle */
      if(!(linep[g1].moves&0x0001)) {	/* Take into account written moves */
	if(!(linep[f1].attackedby.word[Enemy_color]|/* Check attaked squares */
	     linep[g1].attackedby.word[Enemy_color]))
	  if(!(linep[f1].wood.cost|	/* Check occupated squares */
	       linep[g1].wood.cost))
		WRITE_MOVE(movep,line+e1,line+g1,M_CASTLE_RIGHT,CASTLE_PROFIT);
					/* Write move and shift move pointer */
      }
    }
    if(CASTLE[Our_color] & CASTLELEFT ) {	/* Consider left castle */
      if(!(linep[c1].moves&0x0001)) {	/* Take into account written moves */
	if(!(linep[c1].attackedby.word[Enemy_color]|/* Check attaked squares */
	     linep[d1].attackedby.word[Enemy_color]))
	  if(!(linep[b1].wood.cost|	/* Check occupated squares */
	       linep[c1].wood.cost|
	       linep[d1].wood.cost))
		WRITE_MOVE(movep,line+e1,line+c1,M_CASTLE_LEFT,CASTLE_PROFIT);
					/* Write move and shift move pointer */
      }
    }
}

static GEN_FUNCTION gen_promotion	/* Gen promotions */
	       /* Attention! This function doesn't check written moves.
		 Thus it may be called only once. */
{
  SQUARE_NUM from;		/* Square coordinate for move */
  SQUARE_INFO *fromp,*top;/* Pointers to square information */
  MAT_EVAL profit;		/* Profit of writing move */
  WORD pm;			/* Current our piece mask */
  WORD our_att,enemy_att;	/* Piece masks for current square */

    if(Our_color==WHITE_PIECE) {	/* Is our color is white ? */
      from=a7;				/* Initialize square loop */
	  fromp=&SQUARE[a7];		/* Current square information pointer */
      do {				/* Square loop on 7-th line */
	pm=fromp->who_mask.word[WHITE_PIECE];/* Take piece mask */
	if(pm&PIECE_MASKS[PAWN].word[WHITE_PIECE]) {/* Is it white pawn ? */
	  top=fromp+8;			/* Take promotion square */
					  /* information pointer */
	  if(!top->wood.cost) {		/* Is the square empty ? */
	    enemy_att=top->attackedby.word[Enemy_color];/* Enemy attacks of the square */
	    our_att=top->attackedby.word[Our_color];/* Our attacks of the square */
	    profit=
	      move_profit(our_att,enemy_att,PAWN_COST,QUEEN_COST-PAWN_COST,fromp,top);
	    STORE_MOVES(top, pm);		/* Store written move */
	    WRITE_MOVE(movep,from,from+8,M_PROMOTION,profit);
					/* Write move and shift move pointer */
	  }
	      /* Generates capture promotion */
	  if(from!=h7){                       /* Can we attack right? */
	    top=fromp+9;
	    if(top->who_mask.word[BLACK_PIECE]){    /* Is the enemy piece to attack? */
	      enemy_att=top->attackedby.word[Enemy_color];/* Enemy attacks of the square */
	      our_att=top->attackedby.word[Our_color];/* Our attacks of the square */
	      profit=
		move_profit(our_att^pm,enemy_att,PAWN_COST,QUEEN_COST-PAWN_COST+top->wood.cost,fromp,top);
	      STORE_MOVES(top, pm);		/* Store written move */
	      WRITE_MOVE(movep,from,from+9,M_PROMOTION,profit);
	    }
	  }
	  if(from!=a7){                       /* Can we attack left? */
	    top=fromp+7;
	    if(top->who_mask.word[BLACK_PIECE]){    /* Is the enemy piece to attack? */
	      enemy_att=top->attackedby.word[Enemy_color];/* Enemy attacks of the square */
	      our_att=top->attackedby.word[Our_color];/* Our attacks of the square */
	      profit=
		move_profit(our_att^pm,enemy_att,PAWN_COST,QUEEN_COST-PAWN_COST+top->wood.cost,fromp,top);
	      STORE_MOVES(top, pm);		/* Store written move */
	      WRITE_MOVE(movep,from,from+7,M_PROMOTION,profit);
	    }
	  }
	}
      } while(				/* End of 7-th line square loop */
	  fromp++,			/* Take next square information pointer */
	  ++from<=h7			/* Take next square coordinate */
	);				/* Do while coordinate <= h7 */
    }
    else {    /* BLACK_PIECE */			/* Our color is black */
      from=a2;				/* Initialize square loop */
	  fromp=&SQUARE[a2];		/* Current square information pointer */
	  do {				/* Square loop on 2-nd line */
	pm=fromp->who_mask.word[BLACK_PIECE];/* Take piece mask */
	if(pm&PIECE_MASKS[PAWN].word[BLACK_PIECE]) {/* Is it black pawn ? */
	  top=fromp-8;			/* Take promotion square */
					  /* information pointer */
	  if(!top->wood.cost) {		/* Is the square empty ? */
	    enemy_att=top->attackedby.word[Enemy_color];/* Enemy attacks of the square */
	    our_att=top->attackedby.word[Our_color];/* Our attacks of the square */
	    profit=
	      move_profit(our_att,enemy_att,PAWN_COST,QUEEN_COST-PAWN_COST,fromp,top);
	    STORE_MOVES(top, pm);		/* Store written move */
	    WRITE_MOVE(movep,from,from-8,M_PROMOTION,profit);
					/* Write move and shift move pointer */
	  }
	      /* Generates capture promotion. I.Sh. */
	  if(from!=a2){                       /* Can we attack right? */
	    top=fromp-9;
	    if(top->who_mask.word[WHITE_PIECE]){    /* Is the enemy piece to attack? */
	      enemy_att=top->attackedby.word[Enemy_color];/* Enemy attacks of the square */
	      our_att=top->attackedby.word[Our_color];/* Our attacks of the square */
	      profit=
		move_profit(our_att^pm,enemy_att,PAWN_COST,QUEEN_COST-PAWN_COST+top->wood.cost,fromp,top);
	      STORE_MOVES(top, pm);		/* Store written move */
	      WRITE_MOVE(movep,from,from-9,M_PROMOTION,profit);
	    }
	  }
	  if(from!=h2){                       /* Can we attack left? */
	    top=fromp-7;
	    if(top->who_mask.word[WHITE_PIECE]){    /* Is the enemy piece to attack? */
	      enemy_att=top->attackedby.word[Enemy_color];/* Enemy attacks of the square */
	      our_att=top->attackedby.word[Our_color];/* Our attacks of the square */
	      profit=
		move_profit(our_att^pm,enemy_att,PAWN_COST,QUEEN_COST-PAWN_COST+top->wood.cost,fromp,top);
	      STORE_MOVES(top, pm);		/* Store written move */
	      WRITE_MOVE(movep,from,from-7,M_PROMOTION,profit);
	    }
	  }
	}
      } while(				/* End of 2-nd line square loop */
	  fromp++,			/* Take next square information pointer */
	  ++from<=h2			/* Take next square coordinate */
	);				/* Do while coordinate <= h2 */
    }
}
static GEN_FUNCTION gen_check		/* Generate checks */
{
  SQUARE_NUM king;		/* Enemy king square */
  SQUARE_NUM *edgep;		/* Pointer to edge of our king */
  SQUARE_NUM square;		/* Square coordinate for move */
  SQUARE_INFO *squarep;		/* Pointer to square information */
  SQUARE_NUM to;		/* Square coordinate for move */
  SQUARE_INFO *top;		/* Pointer to square information */
  BYTE mask;			/* Avaliable directions mask */
  BYTE m;			/* Current direction mask */
  DIR i;			/* Current direction number */
  int step;			/* Coordinate increment */
  int *stepp;			/* Pointer to current coordinate increment */
  PM *pmp;			/* Our queen_bishop & queen_rook mask pointer */
  WORD pawns;			/* Our pawns mask */

    king=enemy_piece_list[0].where;	/* Enemy king coordinate */
	gen_check_flag=CHECK_PROFIT;

      /* PAWN */			/* Try check by pawns */
    pawns=PIECE_MASKS[PAWN].word[Our_color];	/* Our pawns mask */
    if(Our_color==WHITE_PIECE) {	/* White pawns */
      if(GETLINE(king)>2){
	if(GETFILE(king)!=0) {		/* Left-down */
	  to=king+increment[6]; top=SQUAREP(to);
	  go_to_square(to,top,pawns,0);	/* Capture by pawns */
	  pawn_go_to(to,top);		/* Add vertical pawn moves */
	}
	if(GETFILE(king)!=7) {		/* Right-down */
	  to=king+increment[4]; top=SQUAREP(to);
	  go_to_square(to,top,pawns,0);	/* Capture by pawns */
	  pawn_go_to(to,top);		/* Add vertical pawn moves */
	}
      }
    }
    else /* BLACK_PIECE */ {		/* Black pawns */
      if(GETLINE(king)<5){
	if(GETFILE(king)!=0) {		/* Left-up */
	  to=king+increment[0]; top=SQUAREP(to);
	  go_to_square(to,top,pawns,0);	/* Capture by pawns */
	  pawn_go_to(to,top);		/* Add vertical pawn moves */
	}
        if(GETFILE(king)!=7) {		/* Right-up */
	  to=king+increment[2];  top=SQUAREP(to);
	  go_to_square(to,top,pawns,0);	/* Capture by pawns */
	  pawn_go_to(to,top);		/* Add vertical pawn moves */
        }
      }
    }

      /* KNIGHT */			/* Try check by knights */
					/* Initialize direction loop */
    mask=square_kndirmask[ king ];	/* Get availiable direction mask */
    m='\x01';				/* Current direction mask */
    stepp=&knincrement[0];		/* Current pointer to step */
    do {				/* Direction loop */
      if(m&mask) {			/* Is direction avaliable ? */
	to=king+*stepp;			/* Coordinate */
	top=SQUAREP(to);        /* Its square information pointer */
	if(top->wood.cost) {		/* Is it capture ? */
	  if(top->wood.cost>=mat_deficit)
	    continue;			/* This capture was already written */
	}
	go_to_square(to,top,PIECE_MASKS[KNIGHT].word[Our_color],0);
					/* Go to this squares by knights */
      }
    } while(				/* End of direction loop */
	stepp++,        		/* Take next derection increment pointer */
	m<<=1				/* Take next derection mask */
      );				/* Do while direction mask not empty */

      /* QUEEN ROOK BISHOP */		/* Try check by queens, rooks or bishops */
    pm_queen_bishop.mask=PIECE_MASKS[QUEEN].word[Our_color]| /* Compute our */
			 PIECE_MASKS[BISHOP].word[Our_color];  /* queen_bishop */
    pm_queen_rook.mask  =PIECE_MASKS[QUEEN].word[Our_color]|   /* and queen_rook */
			 PIECE_MASKS[ROOK].word[Our_color];    /* piece mask */
    i=0;				/* Initialize direction loop */
    mask=square_dirmask[king]; 		/* Get availiable direction mask */
    m='\x01';				/* Current direction mask */
	edgep=(SQUARE_NUM *)&SQUAREP(king)->edge[0]; /* Current direction king edge pointer */
		/* Casting is needed for C++ only */
	stepp=&increment[0];		/* Current pointer to step */
    pmp=&pm_queen_bishop;		/* Current pointer to moving pieces mask */
    do {				/* Direction loop */
      if(m&mask) {			/* Is direction avaliable ? */
	square=*edgep;			/* Coordinate for move */
	squarep=SQUAREP(square);	/* Square information pointer */
	if(squarep->wood.color==Our_color) {/* It is our piece ? */
					/* Try to write implicit checks */
	  if(SQUAREP(squarep->edge[i])->who_mask.word[Our_color]&pmp->mask) {
					/* Piece can check king ! */
	    go_from_square(square,squarep,/* Leave this square */
	      ~(m|(m<<HALF(NDIR))|(m>>HALF(NDIR))));
					  /* excluding forward and reverse directions */
	  }
	}
	else {				/* Try write explicit check */
	  go_to_square(square,squarep,pmp->mask,0);
					/* Write moves by queen_bishop */
					    /* or queen-rook */
	}
					/* Initialize squares loop */
	step=*stepp;			/* Coordinate step */
	while((square-=step)!=king) {	/* Loop for squares between */
					  /* king edge and king */
	  squarep-=step;		/* Take next square information pointer */
	  go_to_square(square,squarep,pmp->mask,0);/* Go to this squares by given pieces */
	}
      }
    } while(				/* End of direction loop */
	i++,				/* Take next direction number */
	pmp=pmp->next,			/* Take next moving piece mask pointer */
	edgep++,			/* Take next edge pointer */
	stepp++,			/* Take next step pointer */
	m<<=1				/* Take next derection mask */
      );				/* Do while direction mask not empty */
	gen_check_flag=0;
}

static GEN_FUNCTION gen_all	/* Generate all moves */
				  /* excluding captures en passant & castles */
{
  SQUARE_NUM to;		/* Square coordinate for move */
  SQUARE_INFO *top;		/* Pointer to current square information */

	top=&SQUARE[h8];			/* Initialize squares loop */
    to=h8;				/* First square coordinate */
    do {				/* Squares loop */
      go_to_square(to,top,0xFFFF,0);	/* Write all moves of our pieces to */
					  /* the square */
      if(!top->wood.cost) {		/* Is square empty ? */
	pawn_go_to(to,top);		/* Add pawns vertical moves */
      }
    } while(				/* End of square loop */
	top--,				/* Take next square information pointer */
	to--				/* Take next square coordinate */
      );				/* Do while the square >= a1 */
}
/*-------------------------------------------------------------------*/
			/* Internal generator functions */

int move_cmp(const void *src1, const void *src2)
				/* Compare moves for sorting */
				/* This function cannot have modifier "near" */
{
/*  return(((MOVE *)src2)->profit-((MOVE *)src1)->profit); */
  if(((MOVE *)src2)->profit>((MOVE *)src1)->profit) return 1;
  if(((MOVE *)src2)->profit==((MOVE *)src1)->profit) return 0;
  return -1;
}

static MAT_EVAL near material_deficit(void)
				/* Compute material deficit */
{
    if(G_STATUS&G_FORCED)		/* Status is not NORMAL */
      return(MAT_DEFICIT);
					/* We can capture only */
					  /* more expensive pieces */
    else			/* Status is NORMAL */
      return(-MAT_INFINITY);
					/* We can capture any pieces */
}

static void near en_passant(void)
				/* Generate capture en passant */
{
  SQUARE_NUM to;		/* Square coordinate for move */
  SQUARE_INFO *top;		/* Pointer to current square information */
  PIECE_IN_LIST *our_piecep;/* Pointer to current our piece */
  MAT_EVAL profit;		/* Profit of writing move */
  WORD pm;			/* Current our piece mask */
  WORD att;			/* Our pawn masks for current square */
  WORD our_att,enemy_att;	/* Piece masks for current square */

    to=EN_PASS;				/* Take square en passant */
    if(to!=DUMMY) {			/* Do it exist ? */
      top=SQUAREP(to);
      att=top->attackedby.word[Our_color];/* Our attacks of the square */
      att&=PIECE_MASKS[PAWN].word[Our_color];/* Take into account given piece mask */
      if(att) {				/* Do we attack this square ? */
		enemy_att=top->attackedby.word[Enemy_color];/* Enemy attacks of the square */
		our_att=top->attackedby.word[Our_color];/* Our attacks of the square */
	our_piecep=&our_piece_list[8];	/* Initailize our piece loop */
	pm=0x0100;			/* Our piece mask */
					/* Begin from our first pawn */
	do {				/* Our piece loop */
	  if(att&pm) {			/* Is current our piece attacking square ? */
		profit=
		  move_profit(our_att^pm,enemy_att,PAWN_COST,PAWN_COST,SQUAREP(our_piecep->where),top);
		WRITE_MOVE(movep,our_piecep->where,to,M_EN_PASSANT,profit);
					/* Write move and shift move pointer */
	  }
	} while(			/* End of our piece loop */
	    our_piecep++,		/* Take next our piece pointer */
	    pm<<=1 	 		/* Take next piece mask */
	  );				/* Do while mask not empty */
      }
    }
}
static void near king_escape(SQUARE_NUM from, SQUARE_INFO *fromp, BYTE dir_mask)
				/* King escape from check in */
				  /* given directions without captures. */
{
  SQUARE_INFO *top;		/* Pointer to current square information */
  int *stepp;			/* Coordinate step pointer */
  BYTE mask;			/* Avaliable directions mask */
  BYTE m;			/* Current direction mask */
  MAT_EVAL profit;		/* profit of generated move */

					/* Initialize direction loop */
    mask=square_dirmask[from];		/* Get availiable direction mask */
    mask&=dir_mask;			/* Take into account given directions */
    m='\x01';				/* Current direction mask */
    stepp=&increment[0];		/* Current coordinate increment */
    do {			 	/* Direction loop */
      if(m&mask) {			/* Is direction available ? */
	top=SQUAREP(from+*stepp);		/* Square to king move */
	if(!top->who_mask.word[Our_color]) { /* No our piece on square ? */
	  if(!top->attackedby.word[Enemy_color]) {
					/* Is square not attacked by enemy ? */
	    if(!(0x0001&top->moves)) {	/* Take into account written moves */
		  if((G_STATUS & G_FORCED) && top->wood.cost<mat_deficit) continue;

		  profit=
		    move_profit(0,0,KING_COST,top->wood.cost,fromp,top);
		  STORE_MOVES(top,0x0001);		/* Store written move */
		  WRITE_MOVE(movep,from,from+*stepp,NIL,profit);
				/* Write move and shift move pointer */
	    }
	  }
	}
      }
    } while(				/* End of direction loop */
       stepp++,				/* Take next coordinate increment */
       m<<=1				/* Take next direction mask */
      );				/* Do while direction mask not empty */
}
static void near go_to_square(SQUARE_NUM to, SQUARE_INFO *top, WORD piece_mask, WORD profitable)
				/* Write moves on the square by */
				  /* given set of pieces. */
				/* EXEPTION: it don't write pawns' vertical */
				  /* moves ! */
{
  SQUARE_NUM from;		/* Square coordinate for move */
  SQUARE_INFO *fromp;	/* Pointer to current square information */
  PIECE_IN_LIST *our_piecep;/* Pointer to current our piece */
  MAT_EVAL capt_piece_cost;	/* Cost of captured piece */
  MAT_EVAL our_cost;		/* Cost of our moving piece */
  MAT_EVAL profit;		/* Profit of writing move */
  WORD pm;			/* Current our piece mask */
  WORD our_att,enemy_att;	/* Piece masks for current square */
  WORD att;			/* Mask of pieces which generated moves for */

    if(top->who_mask.word[Our_color]) return; /* Can't capture our pieces */
    our_att=top->attackedby.word[Our_color];/* Our attacks of the square */
    att=our_att & piece_mask;		/* Take into account given piece mask */
    att&=~top->moves;			/* Take into account written moves */
    if(att) {				/* Are there some moves to write ? */
      capt_piece_cost=top->wood.cost;	/* Cost of captured piece */
      enemy_att=top->attackedby.word[Enemy_color]; /* Enemy attacks of the square */
      our_piecep=our_piece_list;	/* Initailize our piece loop */
      pm=0x0001;			/* Our piece mask */
					/* Attempt write king move */
      if(!enemy_att) {			/* Test square for check */
	if(att&pm) {			/* Is current our piece attacking square ? */
	  from=our_piecep->where;	/* From what square we do move */
	  fromp=SQUAREP(from);		/* Pointer to our piece square */
					  /* information */
	  STORE_MOVES(top, pm);		/* Store written move */
	  profit=
	    move_profit(0,0,KING_COST,capt_piece_cost,fromp,top);
	  WRITE_MOVE(movep, from, to, NIL, profit );
					/* Write move and shift move pointer */
	}
      }
      if( att&0xfffe)					/* Write other piece moves */
      while(				/* Our piece loop excluding king */
	  pm<<=1 	 		/* Take next piece mask */
	) {				/* Do while mask not empty */
	our_piecep++; 			/* Take next our piece pointer */
	if(att&pm) { 			/* Is current our piece attacking current enemy ? */
	  from=our_piecep->where;	/* From what square we do move */
	  fromp=SQUAREP(from);		/* Pointer to our piece square */
					  /* information */
	  if(fromp->wood.name==PAWN) {	/* If we move our pawn */
	    if(!capt_piece_cost)	/* Can we capture something ? */
	      continue;			/* Consider next our piece */
	  }

	  our_cost=fromp->wood.cost;	/* Cost of our moving piece */

	  profit=
	    move_profit(our_att^pm,enemy_att,our_cost,capt_piece_cost,fromp,top);

	  if(!profitable||profit>PROFIT_LIMIT/*||to==PREVIOUS->to*/) {
					/* If "profitable" is set write only */
					  /* profitable moves, */
					  /* i.e. profitable captures */
					  /* and capture moving enemy piece */
	    STORE_MOVES(top, pm);		/* Store written move */
	    WRITE_MOVE(movep,from,to,NIL,profit);
				      /* Write move and shift move pointer */
	  }
	}
      }
    }
}

int NEAR mask_to_cost(WORD mask)
{
	if (mask>=256) return(PAWN_COST);
	if (mask>=16) return(KNIGHT_COST);
	if (mask>=4) return(ROOK_COST);
	if (mask>=2) return(QUEEN_COST);
	return(KING_COST);
}

WORD NEAR cost_to_mask(int cost)
{
  if(cost<KNIGHT_COST) return(0);
  if(cost<ROOK_COST) return(PAWN_MASK);
  if(cost<QUEEN_COST) return(BISHOP_KNIGHT_PAWN_MASK);
  return(ROOK_BISHOP_KNIGHT_PAWN_MASK);
}

static void near pawn_go_to(SQUARE_NUM to, SQUARE_INFO *top)
				/* Write pawn vertical moves on square by */
				  /* all our pawns. */
{
  SQUARE_NUM from;		/* Square coordinate for move */
  SQUARE_INFO *fromp;	/* Pointer to current square information */
  WORD pm;			/* Current our piece mask */
  MAT_EVAL profit;
  WORD our_att,enemy_att;

    if(top->wood.cost) return;		/* There is piece on square */
    if(Our_color==WHITE_PIECE) {	/* White pawn */
      if(GETLINE(to)<2) return;
      from=to-8;			/* Where pawn may stand */
					/* Look for pawns jump */
      if(GETLINE(to)==3) {		/* Is it square to jump ? */
	if(!(top-8)->wood.cost) {	/* Is square empty ? */
	  from=to-16;			/* Pawn may jump from here */
	}
      }
    }
    else {     /* BLACK_PIECE */	/* Black pawn */
      if(GETLINE(to)>5) return;
      from=to+8;			/* Where pawn may stand */
					/* Look for pawns jump */
      if(GETLINE(to)==4) {		/* Is it square to jump ? */
	if(!(top+8)->who_mask.dwrd) {	/* Is square empty ? */
	  from=to+16;			/* Pawn may jump from here */
	}
      }
    }
    fromp=SQUAREP(from);		/* Pawn square information */
    if(fromp->wood.name==PAWN) {	/* Do pawn exist in the square ? */
      pm=fromp->who_mask.word[Our_color];/* Take piece mask */
      if(pm) {				/* It is our pawn ? */
	if(!(top->moves&pm)) {		/* Take into account written moves */
	  /* Our attacks of the square */
	  enemy_att=top->attackedby.word[Enemy_color];/* Enemy attacks of the square */
	  our_att=top->attackedby.word[Our_color];/* Our attacks of the square */
	  profit=
	    move_profit(our_att,enemy_att,PAWN_COST,0,fromp,top);
	  STORE_MOVES(top, pm);		/* Store written move */
	  WRITE_MOVE(movep,from,to,NIL,profit);
					/* Write move and shift move pointer */
	}
      }
    }
}

void NEAR go_from_square(SQUARE_NUM from, SQUARE_INFO *fromp, BYTE dir_mask)
				/* Write moves by all pieces from square in */
				  /* given directions. */
{
    switch(fromp->wood.name) {
      case KING:   king_escape(from,fromp,dir_mask); break; /* M.D. */
      case QUEEN:  queen_go_from(from,fromp,dir_mask); break;
      case ROOK:   queen_go_from(from,fromp,dir_mask&'\xAA'); break;
      case BISHOP: queen_go_from(from,fromp,dir_mask&'\x55'); break;
      case KNIGHT: knight_go_from(from,fromp); break;
      default: /* PAWN */
		   pawn_go_from(from,fromp,dir_mask,NIL); break;
    }
}

static void near queen_go_from(SQUARE_NUM from, SQUARE_INFO *fromp, BYTE dir_mask)
				/* Write moves by queen-like piece from */
				  /* square in given directions. */
{
  SQUARE_NUM to;		/* Square coordinate for move */
  SQUARE_INFO *top; 	/* Pointer to current squares information */
  MAT_EVAL profit;		/* Profit of move */
  int step;			/* Coordinate step */
  int *stepp;			/* Coordinate step pointer */
  SQUARE_NUM *edgep;	/* Pointer to edge of our king */
  BYTE mask;			/* Avaliable directions mask */
  BYTE m;			/* Current direction mask */
  WORD pm;			/* Our piece_mask */
  WORD our_att,enemy_att;  /* Attacks masks */

				/* Initialize direction loop */
    mask=square_dirmask[from];	/* Get availiable direction mask */
    mask&=dir_mask;			/* Take into account given directions */
    if(!mask) return;			/* No moves */
    pm=fromp->who_mask.word[Our_color];	/* Our piece mask */
    m='\x01';				/* Current direction mask */
    stepp=&increment[0];		/* Current coordinate increment pointer */
	edgep=(SQUARE_NUM *)&fromp->edge[0];  /* Current edge pointer */
			/* Casting needed for C++ only */
	do {			 	/* Direction loop */
      if((m&mask)) {			/* Is direction are avaliable ? */
					/* Initialize coordinate loop */
					  /* from edge of the square "to" */
					  /* to this square */
	to=*edgep;			/* Current coordinate pointer */
	top=SQUAREP(to);		/* Current square information pointer */
	step=*stepp;			/* Current coordinate increment */
	do {
	 if(!top->who_mask.word[Our_color]) {
					/* No our piece on square */
	  if(!(pm&top->moves)) {	/* Take into account written moves */
		our_att=top->attackedby.word[Our_color];
		enemy_att=top->attackedby.word[Enemy_color];
		profit=
		  move_profit(our_att^pm,enemy_att,fromp->wood.cost,top->wood.cost,fromp,top);
		STORE_MOVES(top, pm);		/* Store written move */
	    WRITE_MOVE(movep,from,to,NIL,profit);
					/* Write move and shift move pointer */
	  }
	 }
	} while(			/* End of coordinate loop */
	    top-=step,			/* Take next square information pointer */
	    (to-=step)!=from		/* Take next square coordinate */
	  );				/* Do while "to!=from" */
      }
    } while(				/* End of direction loop */
	stepp++,			/* Take next coordinate increment */
	edgep++,			/* Take next edge */
	m<<=1				/* Take next direction mask */
      );				/* Do while direction mask not empty */
}

static void near knight_go_from(SQUARE_NUM from, SQUARE_INFO *fromp)
				/* Write moves by knight from square in */
				  /* all directions. */
{
  SQUARE_INFO *top;		/* Pointer to current square information */
  MAT_EVAL profit;		/* Profit of move */
  int *stepp;			/* Coordinate step pointer */
  BYTE mask;			/* Avaliable directions mask */
  BYTE m;			/* Current direction mask */
  WORD pm;			/* Our piece_mask */
  WORD our_att,enemy_att;  /* Attacks masks */

    pm=fromp->who_mask.word[Our_color];	/* Our piece mask */
					/* Initialize direction loop */
	mask=square_kndirmask[from];	/* Get availiable direction mask */
	m='\x01';				/* Current direction mask */
	stepp=&knincrement[0];		/* Current coordinate increment */
	do {			 	/* Direction loop */
	  if(m&mask) {			/* Is direction available ? */
	top=SQUAREP(from+*stepp);		/* Take square information pointer */
	if(!top->who_mask.word[Our_color]) {
					/* No our piece on square */
	  if(!(pm&top->moves)) {	/* Take into account written moves */
		our_att=top->attackedby.word[Our_color];
		enemy_att=top->attackedby.word[Enemy_color];
		profit=
		  move_profit(our_att^pm,enemy_att,KNIGHT_COST,top->wood.cost,fromp,top);
		STORE_MOVES(top, pm);		/* Store written move */
	    WRITE_MOVE(movep,from,from+*stepp,NIL,profit);
					/* Write move and shift move pointer */
	  }
	}
      }
    } while(				/* End of direction loop */
       stepp++,				/* Take next coordinate increment */
       m<<=1				/* Take next direction mask */
      );				/* Do while direction mask not empty */
}
static void near pawn_go_from(SQUARE_NUM from, SQUARE_INFO *fromp, BYTE dir_mask, MOVEFLAGS flag)
				/* Write moves by pawn from square in */
				  /* given directions. */
{
  SQUARE_INFO *top;		/* Pointer to current square information */
  MAT_EVAL profit;		/* Profit of move */
  int *stepp;			/* Coordinate step pointer */
  BYTE mask;			/* Avaliable directions mask */
  BYTE m;			/* Current direction mask */
  WORD pm; 			/* Our piece_mask */

    pm=fromp->who_mask.word[Our_color];	/* Our piece mask */
					/* Initialize direction loop */
    mask=square_dirmask[from];	/* Get availiable direction mask */
    mask&=dir_mask;			/* Take into account given directions */
    if (Our_color==WHITE_PIECE) {
       if (mask & 0x02) { 	/* forward */
	 top=fromp+8;
	 if (!(pm & top->moves))	/* move is available */
	 if (!top->who_mask.dwrd) {
	    gen_pawn_one_move(fromp,top,pm,flag);
	    if(GETLINE(top->mynum)==2) {	/* Is it square to jump ? */
	       top=top+8;		/* jump */
	       if (!top->who_mask.dwrd)
		  gen_pawn_one_move(fromp,top,pm,flag);
	    }
	 }
       }
       if (mask & 0x01) {	/* left */
	 top=fromp+7;
	 if (!(pm & top->moves))	/* move is available */
	 if (top->who_mask.word[Enemy_color])
		  gen_pawn_one_move(fromp,top,pm,flag);
       }
       if (mask & 0x04) {	/* right */
	 top=fromp+9;
	 if (!(pm & top->moves))	/* move is available */
	 if (top->who_mask.word[Enemy_color])
		  gen_pawn_one_move(fromp,top,pm,flag);
       }
    } else {
       if (mask & 0x20) { 	/* forward */
	 top=fromp-8;
	 if (!(pm & top->moves))	/* move is available */
	 if (!top->who_mask.dwrd) {
	    gen_pawn_one_move(fromp,top,pm,flag);
	    if(GETLINE(top->mynum)==5) {	/* Is it square to jump ? */
	       top=top-8;		/* jump */
	       if (!top->who_mask.dwrd)
		  gen_pawn_one_move(fromp,top,pm,flag);
	    }
	 }
       }
       if (mask & 0x40) {	/* left */
	 top=fromp-9;
	 if (!(pm & top->moves))	/* move is available */
	 if (top->who_mask.word[Enemy_color])
		  gen_pawn_one_move(fromp,top,pm,flag);
       }
       if (mask & 0x10) {	/* right */
	 top=fromp-7;
	 if (!(pm & top->moves))	/* move is available */
	 if (top->who_mask.word[Enemy_color])
		  gen_pawn_one_move(fromp,top,pm,flag);
       }
    }
}

static NEAR void gen_pawn_one_move(SQUARE_INFO *fromp,SQUARE_INFO *top,
		WORD pm,MOVEFLAGS flag)
{
  WORD our_att,enemy_att;  /* Attacks masks */
  MAT_EVAL profit;

	  our_att=top->attackedby.word[Our_color];
	  enemy_att=top->attackedby.word[Enemy_color];
	  profit=
		move_profit(our_att^pm,enemy_att,PAWN_COST,top->wood.cost,fromp,top);
	  STORE_MOVES(top, pm);/* Store written move */
	  WRITE_MOVE(movep,fromp->mynum,top->mynum,flag,profit);
}

static GEN_FUNCTION gen_profitable_attack	/* Generate profitable captures */
{
  SQUARE_NUM to;		/* Square coordinate for move */
  SQUARE_INFO *top; 	/* Pointer to square information */
  PIECE_IN_LIST *enemy_piecep;/* Pointer to current enemy piece */
  WORD pm;			/* Current our piece mask */
  WORD avmask;

    enemy_piecep=&enemy_piece_list[1];	/* Skip king */
    pm=0x0002;				/* Enemy piece mask */
    do {				/* Enemy pieces loop */
      to=enemy_piecep->where;		/* Take enemy piece coordinate */
      if(to==DUMMY) continue;		/* Is enemy piece exist in piece list ? */
      top=SQUAREP(to);  		/* Square information pointer */
      if(top->attackedby.word[Enemy_color])
        avmask=cost_to_mask(top->wood.cost);
      else avmask=0xFFFF;
      attack_square(to,avmask);
    } while(				/* End of enemy piece loop */
	enemy_piecep++, 		/* Take next enemy piece pointer */
	pm<<=1 	 			/* Take next piece mask */
      );				/* Do while mask not empty */
}

static GEN_FUNCTION gen_attack_answer	/* Generate profitable captures */
{
  SQUARE_NUM from;		/* Square coordinate for move */
  SQUARE_INFO *fromp;	/* Pointer to square information */
  PIECE_IN_LIST *our_piecep;/* Pointer to current enemy piece */
  WORD pm;			/* Current our piece mask */
  WORD en_mask;
  WORD our_mask;
  PROFIT_SQ *prof_sqp;

//-----------------  Find 2 biggest PROFITS instead qsort in set_profitable_attack()
	  next_profit=0;
	 curr_profit=0;
	 for(prof_sqp=PROFITS[Our_color];prof_sqp->square!=DUMMY ;prof_sqp++){
	   if( prof_sqp->profit >= curr_profit ){
		  next_profit=curr_profit;
		  curr_profit=prof_sqp->profit;
	   }else if( prof_sqp->profit > next_profit) prof_sqp->profit=next_profit;
	 }
//-----------  A.S. 19.11.91

/*    for(prof_sqp=PROFITS[Our_color];prof_sqp->square!=DUMMY &&
	prof_sqp->profit==;prof_sqp++){*/
//    curr_profit=PROFITS[Our_color][0].profit;
    if(	(curr_profit-(next_profit=PROFITS[Our_color][1].profit))>0) {
      defence_flag=1;
      defence_from_attack( PROFITS[Our_color][0].square );
      defence_flag=0;
    }


/*    our_piecep=&our_piece_list[1];	/* Skip king */
    pm=0x0002;				/* Our piece mask */
    do {				/* Our pieces loop */
      from=our_piecep->where;		/* Take our piece coordinate */
      if(from==DUMMY) continue;		/* Is our piece exist in piece list ? */
      fromp=SQUAREP(from);  		/* Square information pointer */
      our_mask=fromp->attackedby.word[Our_color];
      en_mask=fromp->attackedby.word[Enemy_color];

      if(en_mask &&                                    /* Attacked */
         (!our_mask ||                                 /* Not defenced */
          mask_to_cost(en_mask) < fromp->wood.cost))   /* Profit attack */
        defence_from_attack(from);
    } while(				/* End of enemy piece loop */
	our_piecep++, 		/* Take next enemy piece pointer */
	pm<<=1 	 			/* Take next piece mask */
      );				/* Do while mask not empty */
*/
}

static void near attack_square(SQUARE_NUM en_sq, WORD avmask)		/* Generate checks */
{
  SQUARE_NUM *edgep;	/* Pointer to edge of our king */
  SQUARE_NUM square;		/* Square coordinate for move */
  SQUARE_INFO *squarep;	/* Pointer to square information */
  SQUARE_NUM to;		/* Square coordinate for move */
  SQUARE_INFO *top;		/* Pointer to square information */
  BYTE mask;			/* Avaliable directions mask */
  BYTE m;			/* Current direction mask */
  DIR i;			/* Current direction number */
  int step;			/* Coordinate increment */
  int *stepp;			/* Pointer to current coordinate increment */
  PM *pmp;			/* Our queen_bishop & queen_rook mask pointer */
  WORD pawns;			/* Our pawns mask */

    if(!avmask) return;
      /* PAWN */			/* Try check by pawns */
    pawns=PIECE_MASKS[PAWN].word[Our_color];	/* Our pawns mask */
    if(Our_color==WHITE_PIECE) {	/* White pawns */
      if(GETLINE(en_sq)>2){
        if(GETFILE(en_sq)!=0) {		/* Left-down */
	  to=en_sq+increment[6]; top=SQUAREP(to);
	  go_to_square(to,top,pawns,0);	/* Capture by pawns */
	  pawn_go_to(to,top);		/* Add vertical pawn moves */
        }
        if(GETFILE(en_sq)!=7) {		/* Right-down */
	  to=en_sq+increment[4]; top=SQUAREP(to);
	  go_to_square(to,top,pawns,0);	/* Capture by pawns */
	  pawn_go_to(to,top);		/* Add vertical pawn moves */
        }
      }
    }
    else /* BLACK_PIECE */ {		/* Black pawns */
      if(GETLINE(en_sq)<5){ 
        if(GETFILE(en_sq)!=0) {		/* Left-up */
	  to=en_sq+increment[0]; top=SQUAREP(to);
	  go_to_square(to,top,pawns,0);	/* Capture by pawns */
	  pawn_go_to(to,top);		/* Add vertical pawn moves */
        }
        if(GETFILE(en_sq)!=7) {		/* Right-up */
	  to=en_sq+increment[2];  top=SQUAREP(to);
	  go_to_square(to,top,pawns,0);	/* Capture by pawns */
	  pawn_go_to(to,top);		/* Add vertical pawn moves */
        }
      }
    }

    if(!(avmask&KNIGHT_MASK)) return;

      /* KNIGHT */			/* Try check by knights */
					/* Initialize direction loop */
    mask=square_kndirmask[en_sq];	/* Get availiable direction mask */
    m='\x01';				/* Current direction mask */
    stepp=&knincrement[0];		/* Current pointer to step */
    do {				/* Direction loop */
      if(m&mask) {			/* Is direction avaliable ? */
	to=en_sq+*stepp;			/* Coordinate */
	top=SQUAREP(to);            /* Its square information pointer */
	if(top->wood.cost) {		/* Is it capture ? */
	  if(top->wood.cost>=mat_deficit)
	    continue;			/* This capture was already written */
	}
	go_to_square(to,top,PIECE_MASKS[KNIGHT].word[Our_color],0);
					/* Go to this squares by knights */
      }
    } while(				/* End of direction loop */
	stepp++,        		/* Take next derection increment pointer */
	m<<=1				/* Take next derection mask */
      );				/* Do while direction mask not empty */

      /* QUEEN ROOK BISHOP */		/* Try check by queens, rooks or bishops */
    pm_queen_bishop.mask=PIECE_MASKS[QUEEN].word[Our_color]| /* Compute our */
			 PIECE_MASKS[BISHOP].word[Our_color];  /* queen_bishop */
    pm_queen_rook.mask  =PIECE_MASKS[QUEEN].word[Our_color]|   /* and queen_rook */
			 PIECE_MASKS[ROOK].word[Our_color];    /* piece mask */
    i=0;				/* Initialize direction loop */
    mask=square_dirmask[en_sq];	/* Get availiable direction mask */
    m='\x01';				/* Current direction mask */
	edgep=( SQUARE_NUM *)&SQUAREP(en_sq)->edge[0];/* Current direction king edge pointer */
			/* Casting needed for C++ only */
	stepp=&increment[0];		/* Current pointer to step */
    pmp=&pm_queen_bishop;		/* Current pointer to moving pieces mask */
    do {				/* Direction loop */
      if(m&mask) {			/* Is direction avaliable ? */
	square=*edgep;			/* Coordinate for move */
	squarep=SQUAREP(square);	/* Square information pointer */
	if(squarep->wood.color==Our_color) {/* It is our piece ? */
					/* Try to write implicit checks */
	  if(SQUAREP(squarep->edge[i])->who_mask.word[Our_color]&pmp->mask&avmask) {
					/* Piece can check king ! */
	    go_from_square(square,squarep,/* Leave this square */
	      ~(m|(m<<HALF(NDIR))|(m>>HALF(NDIR))));
					  /* excluding forward and reverse directions */
	  }
	}
	else {				/* Try write explicit check */
	  go_to_square(square,squarep,pmp->mask&avmask,0);
					/* Write moves by queen_bishop */
					    /* or queen-rook */
	}
					/* Initialize squares loop */
	step=*stepp;			/* Coordinate step */
	while((square-=step)!=en_sq) {	/* Loop for squares between */
					  /* king edge and king */
	  squarep-=step;		/* Take next square information pointer */
	  go_to_square(square,squarep,pmp->mask&avmask,0);/* Go to this squares by given pieces */
	}
      }
    } while(				/* End of direction loop */
	i++,				/* Take next direction number */
	pmp=pmp->next,			/* Take next moving piece mask pointer */
	edgep++,			/* Take next edge pointer */
	stepp++,			/* Take next step pointer */
	m<<=1				/* Take next derection mask */
      );				/* Do while direction mask not empty */
}

static void near defence_from_attack(SQUARE_NUM our_sq)	/* Generate answers for check */
{
  SQUARE_INFO *our_sqp;		/* Our king square information */
  SQUARE_NUM to;		/* Square coordinate for move */
  SQUARE_INFO *top;		/* Pointer to current square information */
  PIECE_IN_LIST *enemy_piecep;	/* Pointer to current enemy piece */
  WORD pm;			/* Current our piece mask */
  WORD check;			/* Piece mask for our king square */
  int step;			/* Coordinate step */
  BYTE check_dir;		/* Checking directions */
  int check_counter;		/* Check state indicator */

    our_sqp=SQUAREP(our_sq);		/* Our king square information pointer */
					/* Initialize enemy piece loop */
    check=our_sqp->attackedby.word[Enemy_color];	/* Who attack our king */
    enemy_piecep=enemy_piece_list;	/* Skip king */
    pm=0x0001;				/* Enemy piece mask */
    check_dir=NIL;			/* Check directions mask */
    check_counter=bit_quantity[*((BYTE*)&check)]+
		  bit_quantity[*((BYTE*)&check+1)];
    if(check_counter<2) {		/* Not double check */
      do {				/* Enemy piece loop */
	if(check&pm) {  		/* Is current enemy piece attacking our king ? */
	  to=enemy_piecep->where;	/* Checking piece coordinate */
	  top=SQUAREP(to);		/* Checking piece square information */
	  switch(top->wood.name) {	/* Store check direction */
            case KING:
	    case PAWN:
	    case KNIGHT: step=0;	/* We can't defence king */
			 break; 	/* No unavaliable direction for king */
	    default:     check_dir|=direction[to][our_sq];	/* Check direction mask */
			 step=increment[reverse_dir((byte_to_int(check_dir)))];
					/* Take coordinate increment for king defence */
	  }
	  break;			/* No more check pieces */
	}
      } while(				/* End of enemy piece loop */
	  enemy_piecep++, 		/* Take next enemy piece pointer */
	  pm<<=1 	 			/* Take next piece mask */
	);				/* Do while mask not empty */
      if(top->wood.cost>=mat_deficit)	/* Can we capture ? */
	go_to_square(to,top,~0x0000,0);
		/* Capture checking piece by all our pieces */

      if(!G_STATUS||mat_deficit<=0) {	/* Is status NORMAL ? */
	if(step) {			/* Can we defence of our king ? */
	  while((to-=step)!=our_sq) {	/* Loop for squares between */
					  /* checking piece and king */
	    top-=step;			/* Take next square information pointer */
	    go_to_square(to,top,~0x0001,0);/* Go to this squares by all */
					  /* our pieces excluding king */
	    pawn_go_to(to,top);		/* Add vertical pawn moves */
	  }
	}
      }
    }

    go_from_square(our_sq,our_sqp,0xFF);
}

static GEN_FUNCTION gen_attack_escape   /* If win material escape from attack */
{
SQUARE_NUM sq;
  if(!escattorder_en) return;
  if(MAT_DEFICIT<0 && (sq=PROFITS[Our_color][0].square)!=DUMMY
     && PROFITS[Our_color][1].square==DUMMY )
       go_from_square(sq,SQUAREP(sq),0xFF);
}

static signed char far moveprofit[NCOLOR][NPIECE+1][NSQUARE];
void initmoveprofit(void)
{
PIECE_COLOR color;
PIECE_NAME piece;
SQUARE_NUM sq;

  for(color=0;color<NCOLOR;color++)
  for(piece=BISHOP;piece<=PAWN;piece++)  /* Bishop,Knight and pawn */
  for(sq=0;sq<NSQUARE;sq++){
    if(GETLINE(sq)<6 && GETLINE(sq)>1 &&   /* Central square */
       GETFILE(sq)<6 && GETFILE(sq)>1)
      moveprofit[color][piece][sq]=5;
  } /* Tripled for */
}

#define COST_VALUE (our_cost>>2)

static MAT_EVAL move_profit(WORD our_att,WORD enemy_att,
      MAT_EVAL our_cost, MAT_EVAL capt_piece_cost,
      SQUARE_INFO *fromp,SQUARE_INFO *top)
{
  MAT_EVAL profit=0;
  MAT_EVAL our_lost;
  MAT_EVAL future_mat_deficit=0;
  SQUARE_NUM from;
  int castle;
  SQUARE_INFO *sqp;
  int cur_dir;

  future_mat_deficit=capt_piece_cost-((cascadeflag)? /* Narrow material boards ? */
			MAT_DEFICIT:
		    (BETA(level).material-MATERIAL_EVAL));
			  /* Profit at square 'to' */
/*if((level>=mindepth-1) && (NSLOW>=minnslow-1) && !CHECK){   /* Level is before last */*/
			/* Profit at square 'to' */
  if( G_STATUS&G_FORCED &&                 /* If FV and enemy will not  */
      future_mat_deficit > our_cost ){   /* be able to capture our piece */
	profit=capt_piece_cost<<PROFITSHIFT;
  }
  else{                          /* We will also be captured */
   /* linked itself:   Piece---->From---->To   */
	if( fromp->wood.name != KNIGHT ){ /* Knight can't be linked itself */
	  if( (sqp=SQUAREP(fromp->edge[ cur_dir=byte_to_int(direction
	   [top->mynum][fromp->mynum] ) ] ) )!= fromp ){  /* Edge - other piece */
		if( Our_color==sqp->wood.color ){
		our_att|=sqp->who_mask.word[Our_color] &
			 ((cur_dir & 1 )?QUEEN_ROOK_MASK : QUEEN_BISHOP_MASK);
		}
		else if( Enemy_color==sqp->wood.color ){
		enemy_att|=sqp->who_mask.word[Enemy_color] &
			   ((cur_dir & 1 )?QUEEN_ROOK_MASK : QUEEN_BISHOP_MASK);
		}
	  }
	}/* if linked itself */
	profit=
	  (capt_piece_cost-LOST(our_att,enemy_att,our_cost,our_lost))
      <<PROFITSHIFT;
    if( defence_flag ){
      if( (profit>>PROFITSHIFT) > (-curr_profit) ){
	if( (profit>>PROFITSHIFT) > (-next_profit) )
		profit= (curr_profit-next_profit)<<PROFITSHIFT;
	else    profit+= curr_profit<<PROFITSHIFT;
	return( profit - KING_COST - 1 ); /* defence after capture */
      }
    }
  }

  if(profit>0){                           /* Profitable capture */
    return(profit-COST_VALUE);
  }
  else if(profit==0){
    if(capt_piece_cost){                  /* Exchange */
      if(top==SQUAREP(PREVIOUS->to))       /* Capture last moved piece */
	return((1<<PROFITSHIFT)+COST_VALUE);
      if(MAT_DEFICIT<0/*we_win()*/) return((1<<PROFITSHIFT)+COST_VALUE);
      else return(EXCHANGE_PROFIT);
    }
  }
  else{                                  /* Not profitable */
/*
    if(top==SQUAREP(PREVIOUS->to))       /* Capture last moved piece */
	return((1<<PROFITSHIFT)-COST_VALUE);
    else */ return(profit-COST_VALUE);
  }

  if( CHECK ){                    /* We are checked */
    if( our_cost==KING_COST ){    /* King escape from check */
      castle=( (CASTLE[Our_color] & CASTLELEFT ) ||
	       (CASTLE[Our_color] & CASTLERIGHT ) );
					/* Is castle avaliable ? */
      return((castle) ? -KING_COST :0 );
    }
    else return(-COST_VALUE);   /* Piece_defense from check */
  }
  else{                        /*  We are not checked */
	if(enemy_att) profit-=5; /* Don't produce fv */
	if( G_STATUS==G_NORMAL ||                 /* If FV and enemy will */
	future_mat_deficit <= our_cost ){   /* be able to capture our piece */
	  enemy_att=fromp->attackedby.word[Enemy_color]/*^top->who_mask.word[Enemy_color]*/;
	  our_att  =fromp->attackedby.word[Our_color];
	  if( (our_lost=LOST(our_att,enemy_att,our_cost,our_lost))!=0 ){
	from=fromp->mynum;
	profit+=our_lost<<
	  (((pos_sp-1)->n.square[from].attackedby.word[Enemy_color]<enemy_att ||
		(pos_sp-1)->n.square[from].attackedby.word[Our_color]>our_att)
		 ? NEWATTACKSHIFT : OLDATTACKSHIFT );
	  }
	}
	profit-=COST_VALUE;  /* Small pieces move first */
	profit+=gen_check_flag;  /* Bonus if move is check */
//	profit+=moveprofit[MOVE_COLOR][fromp->wood.name][top->mynum];
	return(profit);
  }
}

void NEAR clear_moves( void )			/* Clear written moves masks */
{
  SQUARE_INFO *top;
    top=&SQUARE[h8];
    do top->moves=0; while((top--)!=&SQUARE[a1]);
}
