/*
	 ͻ
	          Working with debut base file.                         
	         -reading/writeing moves and levels,                    
	         -memory allocation and freeing.                        
	                 programmed  by  Anthon  Dubets                 
	 ͼ
*/

#include <io.h>              /* open().. */
#include <stdio.h>           /* sprintf() */
#include <fcntl.h>           /* open flags */
#include <alloc.h>           /* alloc() */
#include <mem.h>             /* memcpy() */
#include <dir.h>             /* MAXPATH */
#include <dos.h>             /* FA_ARCH */

#include "dbchess.h"
#include "dbase.h"
#include "dbfile.h"
#include "dbproc.h"
#include "dberror.h"

/*** Format of moves in base file ***********************************
+-----------------------------------------+
| Move num |                              |
| -----------------------------------------
| 0...5    | Title " Kaissa opening book "|
| 6        | Zero move points to first    |
|          | move of base                 |
| 7...     | other moves                  |
+-----------------------------------------+
**** Format of moves in base file ***********************************/

#define DEBUG       1  /* flag for compilation */
#define TITLEMOVE   5
#define TITLESIZE   (sizeof( FILEMOVE )*TITLEMOVE )
static char title[TITLESIZE]=" Kaissa Opening Book v.1 \x1a";

#define CHECK_OK( fun )   if( (i=fun)!=OK ) return(i)
static int basefh=ERROR; /* debut base file handle */
enum dberror errorcode;
char *errmessage[]={
" Everythink Ok ",
" File open error. ",
" File close error. ",
" File isn't Kaissa debut library. ",
" Read error in file. ",
" Write error in file. ",
" File position error. ",
" Illegal move in file. ",
" No memory for operation. ",
" Save move error. "
};

static int near readfilemove( long int index, FILEMP fmp );
static int near readmove( long int index, BASEMP bmp );
static int near writefilemove( long int index, FILEMP fmp );
static int near writemove( BASEMP bmp );
static int near rewritemove( BASEMP bmp );
static int near writenewmove( BASEMP bmp );
static int near writenewlevel( BASEMP bmp );
static int near erasemove( BASEMP bmp );
static int near eraselevel( BASEMP bmp );
static int near zeromove( long int index );

int flush( int fh );

static int near errtrap( enum dberror errcode )
{
	return( errorcode=errcode );
}

int openbase( char *srcfile )
{
  char buff[TITLESIZE+1], *cp1, *cp2;
  int i;
	if( basefh!=ERROR )                /* file already open */
		closebase();                   /* close old file */
	basefh=open( srcfile, O_RDWR|O_BINARY ); /* open as binary */
	if( basefh==ERROR )                /* check I/O error */
		return( errtrap(OPEN_ERR) );            /* return Error code */
	cp1=buff; cp2=title;               /* check title of file */
	i=_read( basefh, cp1, TITLESIZE ); /* read title */
	if( i!=TITLESIZE )                 /* check read error */
		return( errtrap(READ_ERR) );            /* error code */
	while( *cp1 )                      /* while not '\0' */
		if( *cp1++ != *cp2++ ){        /* compare titles */
			closebase();
			return( errtrap(FILE_ERR) );        /* file isn't Kaissa debut */
		}
	return(OK);                        /* everything Ok */
}

int filecopy( int destfh, int srcfh )
{
  unsigned int buffsize, nb, i=0;
  unsigned char *bp;
	if( destfh<0 || srcfh<0 ) return(-1);
	buffsize=coreleft()-0x200;    /* 0x200 = 512b - rezerved */
	bp=calloc( 1, buffsize );     /* allocate buffer in main memory */
	if( !bp ) return( errtrap(NOMEM_ERR) ); /* no memory */
	while( !eof( srcfh ) ){       /* while not end of source file */
		nb=read( srcfh, bp, buffsize ); /* read into buffer */
		if( nb==0 ){ i=OK; break; } /* 0 byte readed = end of file */
		if( nb==0xffff ){ i=READ_ERR; break; } /* read error occure */
		if( write( destfh, bp, nb )==ERROR ) /* write error occure */
			{ i=WRITE_ERR; break; }
	}
	flush( destfh );              /* flush the buffer */
	free( bp );                   /* free memory */
	if( i!=OK ) return( errtrap( i ) ); /* return error code */
	else return(OK);
}

int makebackup( char *srcfile )
{
  int infh, outfh, i;
  char destfile[MAXPATH];
  char drive[MAXDRIVE], dir[MAXDIR], name[MAXFILE], ext[MAXEXT];
	fnsplit( srcfile, drive, dir, name, ext );
	fnmerge( destfile, drive, dir, name, ".bak" );
	infh=open( srcfile, O_RDONLY|O_BINARY );
	outfh=_creat( destfile, 0 );
	if( infh==ERROR || outfh==ERROR )
		return( errtrap(OPEN_ERR) );
	if( (i=filecopy( outfh, infh ))!=OK ) return(i);
	close( infh ); close( outfh );
	return( OK );
}

int maketemp( char *srcfile )
{
  int infh, outfh, i;
  char destfile[MAXPATH];
  char drive[MAXDRIVE], dir[MAXDIR], name[MAXFILE], ext[MAXEXT];
	fnsplit( srcfile, drive, dir, name, ext );
	fnmerge( destfile, drive, dir, name, ".TMP" );
	infh=open( srcfile, O_RDONLY|O_BINARY );
	outfh=_creat( destfile, 0 );
	if( infh==ERROR || outfh==ERROR )
		return( errtrap(OPEN_ERR) );
	if( (i=filecopy( outfh, infh ))!=OK ) return(i);
	close( infh ); close( outfh );
	return( OK );
}

int copytemp( char *destfile )
{
  int destfh, i;
	destfh=_creat( destfile, 0 ); /* create/rewrite in binary mode */
	if( destfh==-1 )
		return( errtrap(OPEN_ERR) );
	lseek( basefh, 0L, SEEK_SET );
	if( (i=filecopy( destfh, basefh ))!=OK ) return(i);
	close( destfh );
	return(OK);
}

int creatbasefile( char *srcfile )
{
  int i;
	if( basefh!= ERROR )
		closebase();
	basefh=_creat( srcfile, 0 ); /* create/rewrite in binary mode */
	if( basefh==-1 )
		return( errtrap(OPEN_ERR) );
	i=write( basefh, title, TITLESIZE ); /* write title */
	if( i!=TITLESIZE )
		return( errtrap(WRITE_ERR) );
	return( writemove( &nullmove ) ); /* write null move */
}

int closebase( void )
{
	if( basefh!=ERROR )
		if( _close( basefh ) ) /* close base file */
			return( errtrap(CLOSE_ERR) );    /* check error */
	basefh=ERROR;
	return(OK);
}

int flush( int fh )
{
  int dupfh;
	dupfh=dup( fh );             /* duplicate file handler */
	if( dupfh==ERROR )
		return( errtrap(CLOSE_ERR) );
	close( dupfh );              /* close dup file */
	return(OK);
}

static int near readfilemove( long int index, FILEMP fmp )
{
  int i;
  long int l;

	l=lseek( basefh,                   /* use base file hendler */
		index*sizeof( FILEMOVE )+TITLESIZE, /* index from ? */
		SEEK_SET );                    /* seek from begin */
	if( l==ERROR )                     /* check new file position */
		return( errtrap(POSF_ERR) );            /* return file position error */
	i=_read( basefh, fmp, sizeof( FILEMOVE ) ); /* read move from file */
	if( i!=sizeof( FILEMOVE ) )        /* check number of byte read */
		return( errtrap(READ_ERR) );            /* return error code */
	return( OK );
}

static long int near seekbase( long int index, int fromwhere )
{
  long int l;
	switch( fromwhere ){
	  case SEEK_SET: index=index*sizeof( FILEMOVE )+TITLESIZE; break;
	  case SEEK_CUR: break;
	  case SEEK_END: index=index*sizeof( FILEMOVE ); break;
	  default: return( -1L );
	}
	l=lseek( basefh, index,	fromwhere ); /* seek from given position */
	if( l==ERROR )                     /* check new file position */
		return( ERROR );               /* return file position error */
	l=l-TITLESIZE;
	return( l/sizeof( FILEMOVE ) );
}

static int near writefilemove( long int index, FILEMP fmp )
{
  int i;
  long int l;

	l=lseek( basefh,                   /* use base file handler */
		index*sizeof( FILEMOVE )+TITLESIZE, /* index start from ? */
		SEEK_SET );                    /* seek from begin of file */
	if( l==ERROR )                     /* check new file position */
		return( errtrap(POSF_ERR) );            /* return file position error */
	i=_write( basefh, fmp, sizeof( FILEMOVE ) ); /* write move to file */
	if( i!=sizeof( FILEMOVE ) )        /* check number of byte write */
		return( errtrap(WRITE_ERR) );           /* return error code */
	return(0);
}

static int near readmove( long int index, BASEMP bmp )
{
  int i;
  FILEMOVE filemove, *fmp=&filemove;

	if( (i=readfilemove( index, fmp ))!=OK ) /* read move from file */
		return( i );                   /* error */
	WHO=FWHO; FROM=FFROM; TO=FTO;      /* make fields of BASE MOVE */
	if( bmp!=&nullmove                 /* check move is correct */
	  &&( WHO<KING || WHO>PAWN || FROM<a1 || FROM>h8 || TO<a1 || TO>h8) )
			return( errtrap(MOVE_ERR) );        /* this isn't move */
	CAPT=FCAPT; ISCHECK=FISCHECK;
	LASTMOVE=FLASTMOVE; BESTMOVE=FBESTMOVE;
	FIRSTMOVE=FFIRSTMOVE;
	bmp->hlink=fmp->hloffs + ((long)fmp->hlseg<<16);
	bmp->vlink=fmp->vloffs + ((long)fmp->vlseg<<16);
	bmp->ownlink=index;
	NEXT=NULL;
	EDITFLAG=0;
	return(0);
}

static int near writemove( BASEMP bmp )
{
  int i;
  FILEMOVE filemove, *fmp=&filemove;
  long int index;

	if( !bmp )
		return( errtrap( MOVE_ERR ) );
	FWHO=WHO; FFROM=FROM; FTO=TO;      /* make fields of FILE MOVE */
	FCAPT=CAPT; FISCHECK=ISCHECK;
	FLASTMOVE=LASTMOVE; FBESTMOVE=BESTMOVE;
	FFIRSTMOVE=FIRSTMOVE;
	fmp->hloffs=(WORD)HLINK; fmp->hlseg=(bmp->hlink>>16);
	fmp->vloffs=(WORD)VLINK; fmp->vlseg=(bmp->vlink>>16);
	index=OWNLINK;
	if( (i=writefilemove( index, fmp ))!=OK ) /* write move into file */
		return( i );
	return(0);
}

static int near shiftmove( long int dest, long int src )
{
  int i;
  FILEMOVE fm, *fmp=&fm;

	if( (i=readfilemove( src, fmp ))!=OK ) /* read move from file */
		return( i );                       /* error */
	return( writefilemove( dest, fmp ) );  /* write move into file */
}

static int near zeromove( long int index )
{
  FILEMOVE fm;
	memcpy( &fm, " Erased ", sizeof( FILEMOVE ) );  /* clear file move */
	return( writefilemove( index, &fm ) ); /* write zero move to file */
}

#define BUFFSIZE 20               /* I think, max number of moves on level */
static BASEMOVE buff[ BUFFSIZE ]; /* less then BUFFSIZE=20 */

BASEMP readlevel( long int index )
{
  BASEMP bmp;
  int i;
	bmp=buff; i=0;
	while( i<BUFFSIZE ){
		if( readmove( index, bmp )!=OK ) /* read one move */
			return( NULL );       /* return error code */
		i++;                      /* increment move counter */
		if( LASTMOVE ) break;     /* move is last on level, break */
		index=HLINK;              /* else get index of next move */
		bmp++;                    /* get next position */
	}
	bmp=calloc( i, sizeof(BASEMOVE) ); /* allocate memory for level */
	if( !bmp )                    /* check memory presence */
		{ errtrap(NOMEM_ERR); return( NULL ); } /* no more mem */
	memcpy( bmp, buff, sizeof( BASEMOVE )*i ); /* copy level to new place */
	FIRSTMOVE=1;                  /* mark move as first on level */
	return( bmp );
}

int getmovedump( char *str, long int index )
{
  int i;
  FILEMOVE fm;
  BYTE *bp;
	if( basefh==ERROR )      /* file is closed */
		return( errtrap(OPEN_ERR) );
	if( (i=readfilemove( index, &fm ))!=OK )
		return( i );
	bp=(BYTE *)&fm;
	sprintf( str, " Dump: %02x %02x %02x %02x %02x %02x %02x ",
		*bp,*(bp+1),*(bp+2),*(bp+3),*(bp+4),*(bp+5),*(bp+6) );
	return(0);
}

int readbranch( BASEMP bmp )
{
  int i;
  long int link;
	if( !bmp ) return(OK);       /* */
	while(1){
		if( (link=VLINK)!=0l ){    /* check vert link exist */
            NEXT=readlevel( link ); /* read next level */
			if( !NEXT ) return( errorcode );
		}
		if( LASTMOVE ) break;     /* check end of level */
		bmp++;                    /* get next move */
	}
	return(OK);
}

int freebranch( BASEMP bmp, BASEMP exclude )
{
	if( !bmp ) return(OK);       /* */
	while(1){
		if( bmp!=exclude && NEXT )
			{ freelevel( NEXT ); NEXT=NULL; }
		if( LASTMOVE ) break;
		bmp++;
	}
	return(OK);
}

int freelevel( BASEMP bmp )
{
	if( !bmp ) return(-1);
	free( bmp );
	return(0);
}

int loadbasefile( char *srcfile )
{
  int i;
  BASEMP bmp;
	if( (i=openbase( srcfile ))!=OK ) return( i );
	setstack(0);               /* get first stack */
	resetstack();
	bmp=&nullmove;                     /* get pointer to NullMove */
	setmem( bmp, sizeof( BASEMOVE ), 0 ); /* clear NullMove */
	if( (i=readmove( 0, bmp ))!=OK ) return( i );
	FIRSTMOVE=LASTMOVE=1;              /* NullMove is first and last */
	loadstack();
	nullmove.next=BPCURRENT=BPFIRST;
	BPCURNUM=0;
	return( OK );
}

int loadstack( void )
{
  int i;
  BASEMP bmp;
	bmp=&nullmove;                     /* get pointer to NullMove */
	if( VLINK ){                       /* if first level exist read it */
        BPFIRST=readlevel( VLINK );    /* read first level */
		if( !BPFIRST ) return( errorcode );
/*		if( (i=readbranch( BPFIRST ))!=OK ) return( i );*/
		CHECK_OK( readbranch( BPFIRST ) );
	}
	BPCURRENT=BPFIRST;
	BPCURNUM=0;
	return( OK );
}

int checklevel( BASEMP nodebmp )
{
  register BASEMP bmp;
  int i;
	bmp=nodebmp;
	if( !( EDITFLAG & SAVELEVEL ) )    /* nothing to do */
		return( OK );
	while(1){
		if( EDITFLAG & REWRITE )   /* overwrite current move */
			if( (i=rewritemove( bmp ))!=OK ) return(i);
		if( EDITFLAG & ADDMOVE  )   /* add current move to end of level */
			if( (i=writenewmove( bmp ))!=OK ) return(i);
		if( EDITFLAG & ADDLEVEL )   /* add new level from current move */
			if( (i=writenewlevel( bmp ))!=OK ) return(i);
		if( EDITFLAG & DELLEVEL )   /* del level from current move */
			if( (i=eraselevel( bmp ))!=OK ) return(i);
		if( EDITFLAG & DELMOVE  )   /* delete current move */
			if( (i=erasemove( bmp ))!=OK ) return(i);
		if( LASTMOVE ) break;
		bmp++;
	}
	clearmask( nodebmp->flags, SAVELEVEL );
	return( OK );
}

int savebasefile( void )
{
  BASEPOS *bpp;
  BASEMP bmp;
  int i;

	if( (i=checklevel( &nullmove ))!=OK ) return(i);
	bpp=stkinfo[0].bp;
    while(1){
		bmp=bpp->first;
		if( bmp==NULL )      /* no moves on this level */
			break;
		if( (i=checklevel( bmp ))!=OK )    /* check current level */
			return(i);
		while(1){                          /* check all branches */
			if( NEXT )
				if( (i=checklevel( NEXT ))!=OK ) return(i);
			if( LASTMOVE ) break;
			bmp++;
		}
/*
		if( bpp->current->next==NULL )
			break;
*/
		bpp++;
	}
	flush( basefh );
	return( OK );
}

static int near rewritemove( BASEMP bmp )
{
  int i;
#if DEBUG
  char str[80];  /* debug */
	sprintfmove( str, " Re-write move: %m", bmp );
	replyline( str );
#endif
	if( (i=writemove( bmp ))!=OK ) return(i);
	clearmask( EDITFLAG, REWRITE );
	return(i);
}

static int near erasemove( BASEMP bmp )
{
  int i;
#if DEBUG
  char str[80];  /* debug */
	sprintfmove( str, " Delete move: %m", bmp );
	replyline( str );
#endif
	if( FIRSTMOVE && LASTMOVE )   /* only one move - must be erased */
		return( errtrap( SAVE_ERR ) );          /* as whole level */
	if( LASTMOVE ){
		i=zeromove( OWNLINK );    /* clear this move in file */
		if( i ) return(i);
		bmp--;                    /* get previouse move */
		LASTMOVE=1;               /* now it will be last on level */
		HLINK=0L;                 /* clear horiz link */
		i=writemove( bmp );       /* rewrite it in file */
/*???*/	LASTMOVE=0; /* restore */
		if( i ) return(i);
		return(OK);
	}                             /* */
	i=shiftmove( OWNLINK, HLINK );/* re-write next move to place of current */
	(bmp+1)->ownlink=OWNLINK;        /* */
	if( i ) return(i);
	i=zeromove( HLINK );          /* clear next move */
	return(i);
}

static int near eraselevel( BASEMP bmp )
{
  int i;
#if DEBUG
  char str[80];  /* debug */
	sprintfmove( str, " Delete level from: %m", bmp );
	replyline( str );
#endif
	bmp->vlink=0;                 /* only break the link with next level */
	i=writemove( bmp );           /* write move to file */
	return(i);
}

static int near writenewmove( BASEMP bmp )
{
  long int index;
  int i;
#if DEBUG
  char str[80];  /* debug */
	sprintfmove( str, " Write move: %m", bmp );
	replyline( str );
#endif
	index=seekbase( 0L, SEEK_END ); /* find last move index */
	if( index==ERROR )
		return( errtrap( POSF_ERR) );
	OWNLINK=index;                /* save own number */
	VLINK=0L;                     /* init vert link */
	HLINK=0L;                     /* init hor link */
	i=writemove( bmp );           /* add to file */
	if( i ) return(i);
	(bmp-1)->hlink=index;         /* link prev. move with current */
	i=writemove( bmp-1 );          /* correct previouse move */
	if( i ) return(i);
	clearmask( EDITFLAG, ADDMOVE );
	return(OK);
}

static int near writenewlevel( BASEMP nodebmp )
{
  BASEMP bmp;
  long int index;
  int i;
#if DEBUG
  char str[80];  /* debug */
	sprintfmove( str, " Write level from: %m", nodebmp );
	replyline( str );
#endif
	index=seekbase( 0L, SEEK_END ); /* find last move index */
	if( index==ERROR )
		return( errtrap(POSF_ERR) );
/*	if( nodebmp!=&nullmove ){ nullmove exist in base file */
	nodebmp->vlink=index;             /* link prev. move with current */
	i=writemove( nodebmp );           /* correct previouse move */
	if( i ) return(i);
/*	}*/
	bmp=nodebmp->next;                /* get first move on new level */
	while(1){
#if DEBUG
		sprintfmove( str, "    write: %m", bmp );
		replyline( str );
#endif
		OWNLINK=index;                /* save own number */
		VLINK=0L;                     /* init vert link */
		if( !LASTMOVE ) HLINK=index+1;
		else            HLINK=0L;     /* init hor link */
		i=writemove( bmp );           /* add to file */
		if( i ) return(i);
		clearmask( EDITFLAG, ADDMOVE ); /* clear flags ADDMOVE and SAVELEVEL */
		if( LASTMOVE ) break;
		bmp++; index++;
	}
	clearmask( nodebmp->flags, ADDLEVEL );
	return(OK);
}
